/**
 * @module jutils
 */


var CLICK_EVENT = document.createEvent('Event');
CLICK_EVENT.initEvent('click', true, true); //can bubble, and is cancellable
/*
 * ============================================================================================================
 *    				                            logging (old func)
 * ============================================================================================================
 */

/**

 *  @description
 *	This is an available debugging function that updates an HTML
 *	DOM element with an ID of 'log'.  It is useful during developing
 *	and debugging pages that contain JavaScript to trace the flow of
 *	the code.
 *
 *  @example
 *	Manually add a DIV <div id='log' style='location...> This specifies where
 *	on the Luxe's display to show the debugging information.
 *	Then call ..
 *
 * @private
 *
 */
var logs = {
    log: function (content) {
        var log = document.getElementById('log');
        this.logHTML(log, content);
    },
    ID1: function (content) {
        var log = document.getElementById('ID1');
        this.logHTML(log, content);
    },
    logHTML: function (log, content) {
        if (!log) {
            return;
        }
        return log.innerHTML += ('<pre>' + content + '');
    }
};

/*
 * ============================================================================================================
 *    								   	           app configs
 * ============================================================================================================
 */

/**

 *  @description
 *	This function displays home page
 *
 * @instance
 *
 */
function goToHomeApp() {
    tools.goToHomeApp();
}

/*
 * ============================================================================================================
 *    				               todo helper functions, review for deletion
 * ============================================================================================================
 */

/**

 *  @description
 *  TODO missing description
 *
 *  @deprecated Since version 1.5.11. Will be deleted in version 1.6. Use getControlsDOMByName instead.
 *
 * @instance
 *
 */
function getControlByName(name) {
    logger.debug('getControlByName: name={}', name);
    var elements = document.querySelectorAll('[eq-name="' + name + '"]');
    if (!elements || !elements.length) {
        logger.error('getControlByName: There is no control with name={}', name);
    }
    return elements && elements.length ? elements : undefined;
}

/**

 *  @description
 *	This function return control DOM elements by name
 *
 *  @param {string} name - control name
 *
 *  @example
 *  getControlDOMByName('someControlName')
 *
 * @instance
 *
 */
function getControlDOMByName(name) {
    logger.debug('getControlDOMByName: name={}', name);
    var elements = document.querySelectorAll('[eq-name="' + name + '"]');
    if (elements && elements.length > 1) {
        logger.warn("getControlDOMByName: More then one controls with name='{}' exist on page", name);
    } else if (!elements || !elements.length) {
        logger.error('getControlDOMByName: There is no control with name={}', name);
    }
    return elements && elements.length ? elements[0] : undefined;
}

/**

 *  @description
 *	This function return controls DOM elements by name
 *
 *  @param {string} name - control name
 *
 *  @example
 *  getControlsDOMByName('someControlName')
 *
 * @instance
 *
 */
function getControlsDOMByName(name) {
    logger.debug('getControlsDOMByName: name={}', name);
    var elements = document.querySelectorAll('[eq-name="' + name + '"]');
    if (!elements || !elements.length) {
        logger.error('getControlsDOMByName: There is no control with name={}', name);
    }
    return elements && elements.length ? elements : undefined;
}

/**

 *  @description
 *	Scrollable area control functionality.
 *
 * @private
 *
 */
var scrollableArea = {
    options: {},
    _scroll: function (controlId, areaType, rowsPerTouch, up) {
        var options = this.options[controlId];
        options.rowsPerTouch = rowsPerTouch;
        if (areaType === 'table') {
            var scrollableAreaId = '#' + controlId + ' table tbody';
            var scrollableAreaEl = document.querySelector(scrollableAreaId);
            var offset = rowsPerTouch * document.querySelector(scrollableAreaId + ' tr:first-child').clientHeight;
            if (up) {
                offset = -offset;
            }
            scrollableAreaEl.scrollTop = scrollableAreaEl.scrollTop + offset;
            this._enableScrollByOffset(controlId);
        } else if (areaType === 'buttons') {
            options.focusedIdx = up ? options.focusedIdx - rowsPerTouch : options.focusedIdx + rowsPerTouch;
            this._showButtons(controlId, options.focusedIdx);
            this.__enableScrollByIndex(controlId, up);
        } else if (areaType === 'checkbox') {
            options.focusedIdx = up ? options.focusedIdx - rowsPerTouch : options.focusedIdx + rowsPerTouch;
            this._showCheckboxes(controlId, options.focusedIdx);
            this.__enableScrollByIndex(controlId, up);
        } else if (areaType === 'text') {
            var scrollableAreaId = '#' + controlId + ' .equ-text';
            var scrollableAreaEl = document.querySelector(scrollableAreaId);
            var lineHeight = window.getComputedStyle(scrollableAreaEl).getPropertyValue('line-height');
            var offset = rowsPerTouch * parseInt(lineHeight.replace('px', ''));
            if (up) {
                offset = -offset;
            }
            scrollableAreaEl.scrollTop = scrollableAreaEl.scrollTop + offset;
            this._enableScrollByOffset(controlId);
        }
    },
    _enableScroll: function (controlId, visibility, up) {
        var options = this.options[controlId];
        if (up) {
            options.scrollUpVisible = visibility;
        } else {
            options.scrollDownVisible = visibility;
        }
        //toggle scroll up/down buttons visibility only if parent container is visible - hidden variable
        if (document.getElementById(controlId).style.visibility === 'visible') {
            var $scrollEl = up ? options.scroll.$scrollBtnUp : options.scroll.$scrollBtnDown;
            $scrollEl.style.visibility = visibility ? 'visible' : 'hidden';
            var controlName = up ? options.scroll.controlNameUp : options.scroll.controlNameDown;
            enableControl(controlName, visibility);
        }
    },
    _enableScrollByOffset: function (controlId, init) {
        var options = this.options[controlId];
        var scrollableAreaEl = options.areaType === 'table' ? document.querySelector("#" + controlId + " table tbody") : document.querySelector("#" + controlId + " .equ-text");
        var visibleAreaHeight = scrollableAreaEl.clientHeight;
        var offset = undefined;
        if (options.areaType === 'table') {
            var scrollableAreaElInstance = scrollableAreaEl.querySelector("tr:first-child");
            if (scrollableAreaElInstance){
                offset = scrollableAreaElInstance.clientHeight;                
            } else {
                offset = 0;
            }
        } else {
            var lineHeight = window.getComputedStyle(scrollableAreaEl).getPropertyValue('line-height');
            offset = parseInt(lineHeight.replace('px', ''));
        }
        offset = options.rowsPerTouch ? options.rowsPerTouch * offset : undefined;
        var scrollHeight = scrollableAreaEl.scrollHeight;
        var scrollTop = scrollableAreaEl.scrollTop;
        if (scrollHeight > visibleAreaHeight) {
            if (scrollTop == 0) {
                this._enableScroll(controlId, true);
                this._enableScroll(controlId, false, true);
            } else if (scrollHeight == scrollTop + visibleAreaHeight) {
                this._enableScroll(controlId, false);
                this._enableScroll(controlId, true, true);
                executeCallBackFunction("onScrolledToBottom", controlId, options.data);
            } else {
                this._enableScroll(controlId, true);
                this._enableScroll(controlId, true, true);
            }
        } else {
            this._enableScroll(controlId, false, true);
            this._enableScroll(controlId, false);
        }
    },
    __enableScrollByIndex: function (controlId, up) {
        var options = this.options[controlId];
        if (options.visibleItems < options.data.length) {
            if (options.focusedIdx > 0) {
                this._enableScroll(controlId, true, true);
            } else if (options.focusedIdx <= 0) {
                this._enableScroll(controlId, false, true);
            }
            if (options.focusedIdx + options.visibleItems >= options.data.length) {
                this._enableScroll(controlId, false);
                executeCallBackFunction("onScrolledToBottom", controlId, options.data);
            } else if (options.focusedIdx + options.visibleItems < options.data.length) {
                this._enableScroll(controlId, true);
            }
        } else {
            this._enableScroll(controlId, false, true);
            this._enableScroll(controlId, false);
        }
    },
    _buttonClickFn: function ($button, buttonData) {
        return function () {
            $button.classList.remove('equ-button-selected');
            void $button.offsetLeft;
            void $button.offsetTop;
            $button.classList.add('equ-button-selected');
            var handler = function (e) {
                if (buttonData && buttonData.callbackFn && typeof window[buttonData.callbackFn] === "function") {
                    window[buttonData.callbackFn](buttonData);
                }
                $button.removeEventListener('animationend', handler);
                $button.removeEventListener('webkitAnimationEnd', handler);
            };
            $button.addEventListener("animationend", handler);
            $button.addEventListener("webkitAnimationEnd", handler);
        };
    },
    _showButtons: function (controlId) {
        var options = this.options[controlId];
        var data = options.data;
        var startIdx = options.focusedIdx;
        var endIdx = options.focusedIdx + options.visibleItems;
        if (startIdx + options.visibleItems > data.length) {
            endIdx = data.length;
        }
        if (startIdx < 0) {
            startIdx = 0;
            endIdx = options.visibleItems;
        }

        var displayData = data.slice(startIdx, endIdx);
        var scrollableAreaEl = document.getElementById(controlId);
        var scrollableAreaElDivs = scrollableAreaEl.querySelectorAll(".equ-button div");
        for (var i = 0; i < scrollableAreaElDivs.length; i++) {
            scrollableAreaElDivs[i].style.backgroundImage = 'none';
        }
        //start workaround for $("#" + controlId).find(".equ-button div").empty().off('click');
        var equButtons = scrollableAreaEl.querySelectorAll(".equ-button");
        for (var i = 0; i < equButtons.length; i++) {
            equButtons[i].style.display = 'none';
        }
        options._listeners = options._listeners || [];
        for (var idx = 0; idx < options._listeners.length; idx++) {
            var el = options._listeners[idx].element;
            el.removeEventListener('click', options._listeners[idx].listener);
        }
        options._listeners = [];
        //end

        for (idx = 0; idx < options.visibleItems; idx++) {
            if (idx < data.length) {
                var buttonData = displayData[idx];
                if(buttonData) {
                    var $button = scrollableAreaEl.querySelectorAll(".equ-button div")[idx];
                    if (buttonData.label) {
                        $button.innerText = buttonData.label;
                    }
                    if (buttonData.imageUrl) {
                        $button.style.backgroundImage = 'url(' + buttonData.imageUrl + ')';
                    }
                    var listener = this._buttonClickFn($button, buttonData);
                    options._listeners.push({element: $button, listener: listener});
                    $button.addEventListener('click', listener);
                    scrollableAreaEl.querySelectorAll(".equ-button")[idx].style.display = 'block';
                    enableControl(controlId + '-btn-' + idx, true);
                } else {
                    enableControl(controlId + '-btn-' + idx, false);
                }
            } else {
                enableControl(controlId + '-btn-' + idx, false);
            }
        }
    },
    _checkboxClickFn: function (idx, itemData, options) {
        return function () {
            itemData.isChecked = !itemData.isChecked;
            setControlValue(options.controlId + '-chbx-' + idx, itemData.isChecked ? 'checked' : 'unchecked');
            if (itemData.callbackFn && typeof window[itemData.callbackFn] === "function"){
                window[itemData.callbackFn](itemData)
            }
            if (options.changeCallbackFn && typeof window[options.changeCallbackFn] === "function"){
                window[options.changeCallbackFn](options.data);
            }
        };
    },
    _showCheckboxes: function(controlId) {
        var options = this.options[controlId];
        var data = options.data;
        var startIdx = options.focusedIdx;
        var endIdx = options.focusedIdx + options.visibleItems;
        if (startIdx + options.visibleItems > data.length) {
            endIdx = data.length;
        }
        if (startIdx < 0) {
            startIdx = 0;
            endIdx = options.visibleItems;
        }

        var displayData = data.slice(startIdx, endIdx);
        var scrollableAreaEl = document.getElementById(controlId);
        var scrollableAreaItems = scrollableAreaEl.querySelectorAll(".equ-scrollable-checkbox-item");
        for (var i = 0; i < scrollableAreaItems.length; i++) {
            scrollableAreaItems[i].style.backgroundImage = 'none';
        }
        //start workaround for $("#" + controlId).find(".equ-button div").empty().off('click');
        for (var i = 0; i < scrollableAreaItems.length; i++) {
            scrollableAreaItems[i].style.display = 'none';
        }
        options._listeners = options._listeners || [];
        for (var idx = 0; idx < options._listeners.length; idx++) {
            var el = options._listeners[idx].element;
            el.removeEventListener('click', options._listeners[idx].listener);
        }
        options._listeners = [];
        //end

        for (idx = 0; idx < options.visibleItems; idx++) {
            if (idx < data.length) {
                var itemData = displayData[idx];
                if(itemData) {
                    var $chkboxContainer = scrollableAreaEl.querySelectorAll(".equ-scrollable-checkbox-item")[idx];
                    if (itemData.label) {
                        $chkboxContainer.querySelector('.equ-scrollable-checkbox-label').innerText = itemData.label;
                    }
                    if (itemData.imageUrl) {
                        $chkboxContainer.style.backgroundImage = 'url(' + itemData.imageUrl + ')';
                    }
                    var listener = this._checkboxClickFn(idx, itemData, options);
                    options._listeners.push({element: $chkboxContainer, listener: listener});
                    $chkboxContainer.addEventListener('click', listener);
                    $chkboxContainer.style.display = 'block';
                    enableControl(controlId + '-chbx-' + idx, true);
                    setControlValue(controlId + '-chbx-' + idx, itemData.isChecked ? 'checked' : 'unchecked');
                } else {
                    enableControl(controlId + '-chbx-' + idx, false);
                }
            } else {
                enableControl(controlId + '-chbx-' + idx, false);
            }
        }
    },
    _initScrollableArea: function (controlId) {
        var options = this.options[controlId];
        options.scroll = {
            controlNameUp: controlId + '-up',
            controlNameDown: controlId + '-down',
            $scrollBtnUp: document.querySelector('#' + controlId + ' .equ-scroll-up'),
            $scrollBtnDown: document.querySelector('#' + controlId + ' .equ-scroll-down')
        };
    },
    _reinitScrollableArea: function (options) {
        if (options.areaType === 'table') {
            this.initTable(options.controlName, options.data, options.scrollToBottom);
        } else if (options.areaType === 'buttons') {
            this.initButtons(options.controlName, options.data, options.scrollToBottom);
        } else if (options.areaType === 'text') {
            this.initText(options.controlName, options.data, options.scrollToBottom);
        } else if (options.areaType === 'checkbox') {
            this.initCheckboxes(options.controlName, options.data, options.changeCallbackFn, options.scrollToBottom);
        }
    },
    scrollStart: function (controlId, areaType) {
        var options = this.options[controlId];
        if (areaType === 'table') {
            var scrollableAreaId = '#' + controlId + ' table tbody';
            var scrollableAreaEl = document.querySelector(scrollableAreaId);
            this.rowHeight = document.querySelector(scrollableAreaId + ' tr:first-child').clientHeight
            this.offsetStart = scrollableAreaEl.scrollTop;
        } else if (areaType === 'buttons') {
            var equButton = document.querySelector('#' + controlId + ' .equ-button');
            this.rowHeight = equButton.getBoundingClientRect().height;
            this.focusedIdxStart = options.focusedIdx;
        } else if (areaType === 'checkbox') {
            var equChkbox = document.querySelector('#' + controlId + ' .equ-scrollable-checkbox-item');
            this.rowHeight = equChkbox.getBoundingClientRect().height;
            this.focusedIdxStart = options.focusedIdx;
        } else if (areaType === 'text') {
            var scrollableAreaId = '#' + controlId + ' .equ-text';
            var scrollableAreaEl = document.querySelector(scrollableAreaId);
            this.rowHeight = window.getComputedStyle(scrollableAreaEl).getPropertyValue('line-height').replace('px', '');
            this.offsetStart = scrollableAreaEl.scrollTop;
        }
    },
    scrollBy: function (controlId, areaType, dy) {
        var scrolled = 0;
        var options = this.options[controlId];
        if ((areaType === 'table') || (areaType === 'text')) {
            var scrollableAreaId;
            if (areaType === 'table'){
                scrollableAreaId = '#' + controlId + ' table tbody';
            } else if (areaType === 'text'){
                scrollableAreaId = '#' + controlId + ' .equ-text';
            }
            var scrollableAreaEl = document.querySelector(scrollableAreaId);

            var dyRounded = parseInt(dy / this.rowHeight) * this.rowHeight;

            var newScrollTop = this.offsetStart - dyRounded;
            if (newScrollTop < 0){
                newScrollTop = 0;
            } else if ((newScrollTop + scrollableAreaEl.clientHeight) > scrollableAreaEl.scrollHeight) {
                newScrollTop = scrollableAreaEl.scrollHeight - scrollableAreaEl.clientHeight;
            }
            
            if (scrollableAreaEl.scrollTop < newScrollTop){
                scrolled = 1;
            } else if (scrollableAreaEl.scrollTop > newScrollTop){
                scrolled = -1;
            }
            if (scrolled){
                scrollableAreaEl.scrollTop = newScrollTop;
                this._enableScrollByOffset(controlId);
            }
        } else if ((areaType === 'checkbox') || (areaType === 'buttons')) {
            var focusedIdxDelta = parseInt(dy / this.rowHeight);
            var newFocusedIdx = this.focusedIdxStart - focusedIdxDelta;
            if (newFocusedIdx < 0){
                newFocusedIdx = 0;
            } else if ((newFocusedIdx + options.visibleItems) > options.data.length) {
                newFocusedIdx = options.data.length - options.visibleItems;
            }

            if (options.focusedIdx < newFocusedIdx){
                scrolled = 1;
            } else if (options.focusedIdx > newFocusedIdx){
                scrolled = -1;
            }
            if (scrolled){
                options.focusedIdx = newFocusedIdx;
                if (areaType === 'checkbox'){
                    this._showCheckboxes(controlId, options.focusedIdx);
                } else if (areaType === 'buttons'){
                    this._showButtons(controlId, options.focusedIdx);
                }
                this.__enableScrollByIndex(controlId);
            }
        }
        return scrolled;
    },
    up: function (controlId, areaType, rowsPerTouch) {
        this._scroll(controlId, areaType, rowsPerTouch, true);
    },
    down: function (controlId, areaType, rowsPerTouch) {
        this._scroll(controlId, areaType, rowsPerTouch, false);
    },
    initTable: function (name, data, scrollToBottom) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        if (!this.options[controlId]) {
            this.options[controlId] = {
                controlId: controlId,
                controlName: name,
                areaType: 'table',
                scrollToBottom: scrollToBottom
            };
            this._initScrollableArea(controlId);
        }
        this.options[controlId].data = data;
        var rowTemplate = document.querySelector("#" + controlId + " script[type='text/template']").innerHTML;
        var htmlContent = Mustache.render(rowTemplate, data);
        document.querySelector('#' + controlId + ' table tbody').innerHTML = null;
        document.querySelector('#' + controlId + ' table tbody').innerHTML = htmlContent;
        if (scrollToBottom) {
            var scrollableAreaId = '#' + controlId + ' table tbody';
            document.querySelector(scrollableAreaId).scrollTop = document.querySelector(scrollableAreaId).scrollHeight;
        }
        this._enableScrollByOffset(controlId, true);
    },
    initButtons: function (name, data, scrollToBottom) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        if (!this.options[controlId]) {
            var visibleItems = document.querySelectorAll("#" + controlId + " .equ-button").length;
            this.options[controlId] = {
                controlId: controlId,
                controlName: name,
                areaType: 'buttons',
                focusedIdx: 0,
                visibleItems: visibleItems,
                scrollToBottom: scrollToBottom
            };
            this._initScrollableArea(controlId);
            secure.onSecureInput.connect(function (data) {
                for (var i = 0; i < visibleItems; i++) {
                    var btnId = controlId + '-btn-' + i;
                    var scrollableAreaEl = document.getElementById(controlId);
                    if (btnId in data) {
                        //scrollableAreaEl.querySelectorAll(".equ-button div")[i].click();
                        scrollableAreaEl.querySelectorAll(".equ-button div")[i].dispatchEvent(CLICK_EVENT);
                        break;
                    }
                }
            });
        }

        this.options[controlId].focusedIdx = scrollToBottom ? (data.length - visibleItems > 0 ? data.length - visibleItems : 0) : this.options[controlId].focusedIdx;
        this.options[controlId].data = data;
        this._showButtons(controlId);
        this.__enableScrollByIndex(controlId);
    },
    initText: function (name, htmlText, scrollToBottom) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        if (!this.options[controlId]) {
            this.options[controlId] = {
                controlId: controlId,
                controlName: name,
                areaType: 'text',
                scrollToBottom: scrollToBottom
            };
            this._initScrollableArea(controlId);
        }
        this.options[controlId].data = htmlText;
        var scrollableAreaEl = document.querySelector('#' + controlId + ' .equ-text');
        scrollableAreaEl.style.height = scrollableAreaEl.parentElement.style.minHeight;
        scrollableAreaEl.innerHTML = htmlText;
        if (scrollToBottom) {
            var scrollableAreaId = '#' + controlId + ' .equ-text';
            document.querySelector(scrollableAreaId).scrollTop = document.querySelector(scrollableAreaId).scrollHeight;
        }
        this._enableScrollByOffset(controlId, true);

    },
    initCheckboxes: function (name, data, changeCallbackFn, scrollToBottom) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        if (!this.options[controlId]) {
            var visibleItems = document.querySelectorAll("#" + controlId + " .equ-scrollable-checkbox-item").length;
            this.options[controlId] = {
                controlId: controlId,
                controlName: name,
                areaType: 'checkbox',
                focusedIdx: 0,
                visibleItems: visibleItems,
                changeCallbackFn: changeCallbackFn,
                scrollToBottom: scrollToBottom
            };
            this._initScrollableArea(controlId);
            secure.onSecureInput.connect(function (data) {
                for (var i = 0; i < visibleItems; i++) {
                    var btnId = controlId + '-chbx-' + i;
                    var scrollableAreaEl = document.getElementById(controlId);
                    if (btnId in data) {
                        scrollableAreaEl.querySelectorAll(".equ-scrollable-checkbox-item")[i].dispatchEvent(CLICK_EVENT);
                        break;
                    }
                }
            });
        }
        for (var i = 0; i < data.length; i++) {
            if (!data[i].isChecked){
                data[i].isChecked = false;
            }
        }
        this.options[controlId].focusedIdx = scrollToBottom ? (data.length - visibleItems > 0 ? data.length - visibleItems : 0) : this.options[controlId].focusedIdx;
        this.options[controlId].data = data;
        this._showCheckboxes(controlId);
        this.__enableScrollByIndex(controlId);
    },
    appendData: function (name, newItem) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        var options = this.options[controlId];
        if (options.areaType === 'text') {
            options.data += newItem;
        } else {
            options.data.push(newItem);
        }
        if (options.areaType === 'buttons') {
            options.focusedIdx = options.scrollToBottom ? options.data.length : options.focusedIdx;
            if (options.data.length <= options.visibleItems) {
                enableControl(controlId + '-btn-' + (options.data.length - 1), true);
            }
        } else if (options.areaType === 'checkbox') {
            options.focusedIdx = options.scrollToBottom ? options.data.length : options.focusedIdx;
            if (options.data.length <= options.visibleItems) {
                enableControl(controlId + '-chbx-' + (options.data.length - 1), true);
            }
        }
        this._reinitScrollableArea(options);
    },
    setData: function (name, data) {
        var controlEl = getControlDOMByName(name);
        var controlId = controlEl.id;
        var options = this.options[controlId];
        options.data = data;
        if (options.areaType === 'buttons') {
            options.focusedIdx = options.scrollToBottom ? data.length : 0;
        } else if (options.areaType === 'checkbox') {
            options.focusedIdx = options.scrollToBottom ? data.length : 0;
        }
        this._reinitScrollableArea(options);
    },
    isScrollUpVisible: function(controlId) {
        return this.options[controlId].scrollUpVisible;
    },
    isScrollDownVisible: function(controlId) {
        return this.options[controlId].scrollDownVisible;
    }
};

/**

 *  @description
 *	This function retrieves a variable that is exptected to be a simple
 *  numeric string with 2 implied decimals places( "000000001299" ) and
 *  change the format to zero supressed with 2 decimal places "12.99"
 *
 *  @param {string} varName - name of the number variable
 *  @return {number} amountVal - formatted amount value.
 *
 *  @private
 *  @deprecated
 *
 * @instance
 *
 */
function getAmount(varName) {
    var amountVal;

    if (isTerminal()) {
        amountVal = registry.getValue(varName);
        logger.info("getAmount: varName={}, value={}", varName, amountVal);
    } else {
        $.ajax({
            type: "GET",
            url: jPhoenixSettings.getBaseUrl() + 'variable/get',
            data: varName,
            dataType: "xml",
            async: false,
            success: function (response) {
                var variable = response.getElementsByTagName("variable");
                for (i = 0; i < variable.length; i++) {
                    if (variable[i].innerHTML == "" || variable[i].innerHTML == "false") {
                        amountVal = false;
                    } else {
                        amountVal = variable[i].textContent;
                    }
                }
            }
        })
    }
    if (amountVal) {
        var valLength = amountVal.length;
        var amount = amountVal.substring(0, valLength - 2) + "." + amountVal.substring(valLength - 2);
        amountVal = amount.replace(/^[0]+/g, "");
    }
    return amountVal;
}

/**

 *  @description
 *	This function send log entry to be wrriten in log file
 *	todo remove this (testing purposes)
 *
 *  @param {string} message - message to be logged in console/log file.
 *
 *  @private
 *  @deprecated
 *
 * @instance
 *
 */
function consoleLog(message) {
    $.get('/cgi-bin/sdk/console/log', {message: message});
}

/**

 *  @description
 *	This function replaces the special characters with preferred symbols.
 *
 *  @param {string} str - string to be replaces
 *  @return {string} formatted String
 *
 * @instance
 *
 */
function htmlEscape(str) {
    return String(str)
        .replace(/&/g, '&amp;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}

/**

 *  @description
 *	This function scrolls an item up after adding a new items to the end of a list
 *
 *  @param {string} id - DOM id for update the scroll.
 *
 *  @private
 *
 * @instance
 *
 */
function updateScroll(id) {
    var objDiv = document.getElementById(id);
    objDiv.style.overflow = "hidden";
    objDiv.scrollTop = objDiv.scrollHeight;
}

/**

 *  @description
 *	This functions determines the type of the value to set the variable name.
 *
 *  @param {(string | number | blob)} value - could be string, number or binary large object (blob)
 *  @return {(string | number | blob)} type - could be any of the type
 *
 *  @private
 *  @deprecated
 *
 * @instance
 *
 */
function determineTypeOfValue(value) {
    var type;
    if (typeof value === "number") {
        type = 'integer';
    } else if (typeof value === 'string') {
        type = 'string';
    } else {
        type = 'blob';
    }
    return type;
}

/**

 *  @description
 *	This function converts Uint8Array to String
 *
 *  @param {array} buf - an array of values
 *  @return {string} converted string
 *
 *  @example
 *  var buf = [110,123,141,152,187];
 *  var str = Uint8ArrayToString(buf);
 *
 * @instance
 *
 */
function Uint8ArrayToString(buf) {
    return String.fromCharCode.apply(null, new Uint8Array(buf));
}

/**

 *  @description
 *	This function converts String to Uint8Array
 *
 *  @param {string} str - text string
 *  @return {object} buffer with array
 *
 *  @example
 *  var str = "hello";
 *  var buf = stringToUint8Array(str);
 *
 * @instance
 *
 */
function stringToUint8Array(str) {
    var buf = new Array(str.length);
    var bufView = new Uint8Array(buf);
    for (var i = 0, strLen = str.length; i < strLen; i++) {
        bufView[i] = str.charCodeAt(i);
    }
    return bufView;
}

/**

 *  @description
 *	This function converts String to HexaDecimalASCII String
 *
 *  @param {string} str - text String
 *  @return {string} HexaDecimal ASCII String
 *
 *  @example
 *  var str = "\x01\x02\x03";
 *  var hex = stringToHexaDecimalASCII(str);
 *
 * @instance
 *
 */
function stringToHexaDecimalASCII(str) {
    var hex, i;
    var result = "";
    for (i = 0; i < str.length; i++) {
        hex = str.charCodeAt(i).toString(16);
        result += ("000" + hex).slice(-4);
    }
    return result
}

/**

 *  @description
 *	This function converts HexaDecimalASCII String to String
 *
 *  @param {string} hex - HexaDecimal String
 *  @return {string} text String
 *
 *  @example
 *  var hex = "010203";
 *  var str = hexaDecimalASCIIToString(hex);
 *
 * @instance
 *
 */
function hexaDecimalASCIIToString(hex) {
    var j;
    var hexes = hex.match(/.{1,4}/g) || [];
    var string = "";
    for (j = 0; j < hexes.length; j++) {
        string += String.fromCharCode(parseInt(hexes[j], 16));
    }
    return string;
}

/**

 *  @description
 *	This function converts rgb string to integer value
 *
 *  @param {string} rgbColor - RGB string (example "rgb(255, 62, 109)")
 *  @return {number} number
 *
 *  @example
 *  rgbStringToInt("rgb(255, 62, 109)") = 16727661;  //16727661=0xFF3E6D
 *
 * @instance
 *
 */
function rgbStringToInt(rgbColor) {
    var rgb = rgbColor.split(',');
    var r = parseInt(rgb[0].substring(4)); // skip rgb(
    var g = parseInt(rgb[1]); // this is just g
    var b = parseInt(rgb[2]); // parseInt scraps trailing )

    return r << 16 | g << 8 | b;
}

/**

 *  @description
 *	This function converts integer value to rgb string
 *
 *  @param {number} intColor - (example 16727661)
 *  @return {string} string
 *
 *  @example
 *  intToRgbString(16727661) = "rgb(255, 62, 109)"
 *
 * @instance
 *
 */
function intToRgbString(intColor) {
    var b = intColor & 0xff;
    intColor = intColor >> 8;
    var g = intColor & 0xff;
    intColor = intColor >> 8;
    var r = intColor & 0xff;
    return "rgb(" + r + ", " + g + ", " + b + ")";
}

/**

 *  @description
 *	This function convert the string representation of some XML into a DOM object.
 *
 *  @param {string} xmlString - xml string
 *
 *  @private
 *
 * @instance
 *
 */
function stringToXMLDom(string) {
    var xmlDoc = null;
    if (window.DOMParser) {
        parser = new DOMParser();
        xmlDoc = parser.parseFromString(string.trim(), "text/xml");
    }
    else // Internet Explorer
    {
        xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
        xmlDoc.async = "false";
        xmlDoc.loadXML(string);
    }
    return xmlDoc;
}

/**

 *  @description
 *	This function analyse varText, replace all #prompts with language specific value and all $sysVar with current variable values
 *	and return array of all $sysVars and translated text.
 *
 *  @param {string} langCode - Could be constant, registry variable or javascript variable
 *  @param {string} varText - Text to be translated
 *  @return {object} {text: '...', variables: []}
 *
 *  @example
 *  parsePlaceholderString("$langCode", '"some constant text" $sysVar #prompt')
 *
 *  @private
 *
 * @instance
 */
function parsePlaceholderString(langCode, varText) {
    logger.debug('parsePlaceholderString: langCode={}, varText={}', langCode, varText);
    return _parsePlaceholderString(langCode, varText);
};

function _parsePlaceholderString(langCode, varText, vars) {
    var variables = vars || [];
    if (langCode && langCode.charAt(0) === '$') {
        if (variables.indexOf(langCode.substring(1)) == -1) {
            variables.push(langCode.substring(1));
        }
        var tempLangCode = variableGet(langCode.substring(1));
        if(!tempLangCode) {
            logger.error("Language Code variable = {} does not exist, default language will be used.", langCode);
        }
        langCode = tempLangCode;
    }
    varText = varText.replace(/"(\\"|[^"])*"|([^"]+)/g, function (m, g1, g) {
        if (g) {
            g = g.replace(/([\$|#][^#^$^<^"^\s+]*)/g, function (mVar, gVar) {
                if (gVar && gVar.charAt(0) === '$') {
                    var varName = gVar.substring(1);
                    if (variables.indexOf(varName) == -1) {
                        variables.push(varName);
                    }
                    var varValue = variableGet(varName);
                    if (varValue !== null && (typeof varValue === 'string')) {
                        var translation = _parsePlaceholderString(langCode, varValue, variables);
                        varValue = translation.text;
                    }
                    return varValue !== null ? varValue : '';
                } else if (gVar && gVar.charAt(0) === '#') {
                    var promptId = gVar.substring(1);
                    var promptValue;
                    if(langCode){
                        promptValue = getCurrentAppPrompt(promptId, langCode);
                        if(!promptValue) {
                            promptValue = getCurrentAppDefaultLangPrompt(promptId);
                            if(promptValue){
                                logger.error("Language Code variable = {} does not exit, default language will be used.", langCode);
                            } else {
                                logger.error("promptId = {} does not exist.", promptId);
                            }
                        }
                    } else {
                        promptValue = getCurrentAppDefaultLangPrompt(promptId);
                        if(!promptValue){
                            logger.error("promptId = {} does not exist for default language.", promptId);
                        }
                    }

                    if (promptValue !== null && (typeof promptValue === 'string')) {
                        var translation = _parsePlaceholderString(langCode, promptValue, variables);
                        promptValue = translation.text;
                    }
                    return promptValue || '';
                } else return mVar;
            });
            return g;
        } else {
            m = m.substring(1, m.length-1);
            return m.replace(/\\"/g, '"');
        }
    });
    if (varText) {
        // Replace \n or \r\n (Windows) or \n (Linux/Unix) with </br>
        varText = varText.replace(/\\n|\n|\r\n/gm, "<br/>");
    }
    logger.debug('parsePlaceholderString: value={}', varText);
    return {text: varText, variables: variables};
};

/**

 *  @description
 *	This function return current application name
 *
 * @instance
 *
 */
function getCurrentApp() {
    var currentApp = page.getUrlInfo().application;
    logger.debug('getCurrentApp: name={}', currentApp);
    return currentApp;
}

/**

 *  @description
 *	This function return localized messages by promptId and language code
 *
 *  @param {string} promptId - localized message id
 *  @param {string} langCode - language code
 *  @return {string} localized message
 *
 *  @example
 *  getCurrentAppPrompt('somePromptId', 'fr')
 *
 * @instance
 *
 */
function getCurrentAppPrompt(promptId, langCode) {
    return prompts.getPrompt(getCurrentApp(), promptId, langCode);
}

/**

 *  @description
 *	This function return localized messages by promptId for default language.
 *
 *  @param {string} localized message id
 *  @return {string} localized message
 *
 *  @example
 *  getCurrentAppDefaultLangPrompt('somePromptId')
 *
 * @instance
 *
 */
function getCurrentAppDefaultLangPrompt(promptId) {
    return prompts.getDefaultLangPrompt(getCurrentApp(), promptId);
}

/**

 *  @description
 *	This function sets a new theme for current application.
 *
 *  @param {string} themeName - theme name
 *
 *  @example
 *  setTheme('theme1')
 *
 * @instance
 *
 */
function setTheme(themeName) {
    var appName = getCurrentApp();
    logger.debug("setTheme: applicationName='{}' , themeName='{}'", appName, themeName);
    var params = JSON.stringify({"applicationName": appName, "themeName": themeName, "reloadCurrentPage": true , "cacheClear": true});
    if (appName && themeName) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', '/cgi-bin/platform/theme?applicationName=' + appName, true);
        xhr.send();
        xhr.onload = function () {
            var currentTheme = JSON.parse(this.response).themeName;
            if (themeName !== currentTheme) {
                logger.debug("setTheme: Changing theme to:'{}' from:'{}' ", themeName, currentTheme);
                xhr = new XMLHttpRequest();
                xhr.open('POST', '/cgi-bin/platform/theme', true);
                xhr.send(params);
                xhr.onload = function () {
                    if (this.status != 200) {
                        logger.error("setTheme: Unable to set theme '{}' - {}", themeName, this.response);
                    } else {
                        logger.debug("setTheme: Theme successfully changed to '{}'", themeName);
                        location.reload(true);
                    }
                };
            }
        };
    }
}

/**

 *  @description
 *	This function clears active logins.
 *
 *  @example
 *  clearActiveLogins();
 *
 * @instance
 *
 */
function clearActiveLogins() {
    logger.debug("clearActiveLogins");
    var xhr = new XMLHttpRequest();
    xhr.open('GET', '/cgi-bin/page/clearActiveLogins', true);
    xhr.send();
    xhr.onload = function () {
        if (xhr.readyState === 4) {
            if (xhr.status === 200) {
                logger.debug("clearActiveLogins {}", xhr.responseText);
            } else {
                logger.error("clearActiveLogins {}", xhr.statusText);
            }
        }
    };
}

/**

 *  @description
 *	This function return all query string params as object.
 *
 *  @param {string} url - url with query string params
 *
 *  @example
 *  getQueryStringParams('http://<host>:<port>/cgi-bin/page/show?path=someApp/page.html&param1=...&param2=...')
 *
 * @instance
 *
 */
function getQueryStringParams(url) {
    var params = {};
    if (url) {
        url.replace(
            /[?&]+([^=&]+)=?([^&]*)?/gi,
            function (m, key, value) {
                params[key] = value !== undefined ? value : '';
            }
        );
    }
    return params;
}

/**

 *  @description
 *	This function return query string param value.
 *
 *  @param {string} url - url with query string params
 *  @param {string} paramName - param name
 *
 *  @example
 *  getQueryStringParam('http://<host>:<port>/cgi-bin/page/show?path=someApp/page.html&param1=...&param2=...', 'param2')
 *
 * @instance
 *
 */
function getQueryStringParam(url, paramName) {
    var params = getQueryStringParams(url);
    if (params && paramName) {
        return params[paramName];
    }
    return undefined;
}

var videoControlsLoopState = {};
/**

 *  @description
 *	This function returns function, used as video 'ended' event callback, to count video loops.
 *
 *  @param {string} domId - domId of video control
 *
 *  @example
 *  handleVideoLoopingEndedEvent("ctrl_id")
 *
 * @instance
 *
 */
function handleVideoLoopingEndedEvent(domId) {
    logger.debug("handleVideoLoopingEndedEvent registered for domId={}", domId);
    return function() {
        var loopCount = videoControlsLoopState[domId].loopCount;
        var loopCurrent = videoControlsLoopState[domId].loopCurrent;
        logger.debug("handleVideoLoopingEndedEvent callback called for domId={}; loopCurrent={}, loopCount={}", domId, loopCurrent, loopCount);
        if (loopCurrent < loopCount) {
            var videoSource = document.querySelector('#' + domId + ' video');
            videoSource.play();
            videoControlsLoopState[domId].loopCurrent++;
        } else {
            videoControlsLoopState[domId].loopCurrent = 1;
        }
    }
}

var audioControlsLoopState = {};
/**

 *  @description
 *	This function returns function, used as audio 'ended' event callback, to count audio loops.
 *
 *  @param {string} domId - domId of audio control
 *
 *  @example
 *  handleAudioLoopingEndedEvent("ctrl_id")
 *
 * @instance
 *
 */
function handleAudioLoopingEndedEvent(domId) {
    logger.debug("handleAudioLoopingEndedEvent registered for domId={}", domId);
    return function() {
        var loopCount = audioControlsLoopState[domId].loopCount;
        var loopCurrent = audioControlsLoopState[domId].loopCurrent;
        logger.debug("handleAudioLoopingEndedEvent callback called for domId={}; loopCurrent={}, loopCount={}", domId, loopCurrent, loopCount);
        if (loopCurrent < loopCount) {
            var audioSource = document.querySelector('#' + domId + ' audio');
            audioSource.play();
            audioControlsLoopState[domId].loopCurrent++;
        } else {
            audioControlsLoopState[domId].loopCurrent = 1;
        }
    }
}

/*
 * ============================================================================================================
 *    								       Internal exception handling
 * ============================================================================================================
 */

/**

 *  @description
 *  This function registers a function on window.onerror for handling all the unhandled errors that get thrown inside the application.
 *
 *  The function gets passed one string argument "message" (NOTE: due to  current limitations in existing QT4.9 browser "message" is the only parameter in the window.onerror).
 *
 *  Variable error is constructed using message argument and if onPageError function exists, the error variable is passed to the onPageError function.
 *
 * @instance
 *
 */


function registerWindowOnError(){
    var existingOnError = window.onerror;
    window.onerror = function(message){
        logger.debug("window.onerror: " + message);

        if (existingOnError) existingOnError.apply(this, arguments);

        var error;

        try {
            error = JSON.parse(message);
            if (error.name == "EQError"){
                delete error.name;
            } else {
                throw message;
            }
        } catch(e){
            error = eqErrors.UNKNOWN_ERROR;
            error.errorSpecificData = {message: message};
        }

        if (typeof onPageError === "function"){
            onPageError(error);
        }

        logger.debug("error: {}", JSON.stringify(error));

        return false;
    }
}

/**

 * @description
 *  This constructor function is used for making an object for internal exception handling.
 *  NOTE: This was implemented due to current limitations in existing qt4.9 browser, where "message" is the only parameter in the window.onerror.
 *
 * @private
 *
 */

function EQError(eqError, errorSpecificData){
    this.errorCode = eqError.errorCode;
    this.errorMessage = eqError.errorMessage;
    this.errorSpecificData = errorSpecificData;
    this.name = "EQError"
}

EQError.prototype.toString = function(){
    return JSON.stringify(this);
}
