#!/bin/sh
#==============================================================================
#  Company:
#      Equinox Payments, LLC
#
#  Product:
#      Platform utilities
#      Copyright 2011-2022 Equinox Payments, LLC
#
#  \brief
#      Collect information about platform state.
#
#==============================================================================

########################################################
# Print a header line, to separate sections of the log
#
#   $1 - Header string
#
PrintHeader( )
{
    HEADER_STR=$1

    echo " "
    echo "**********************************************"
    echo "* ${HEADER_STR}"
    echo "**********************************************"
    echo " "
}

########################################################

IS_ROOT=0

PrintHeader "User"
whoami

# Check if root (if not - some option will be omitted)
if [[ `whoami` == "root" ]]; then
   IS_ROOT=1
fi

PrintHeader "Date & time"
date


PrintHeader "RTC (hwclock)"
cat /proc/driver/rtc

PrintHeader "Mode"
value=`cat /sys/kernel/secure/state`
if [ "$value" == 0 ]; then
echo 'Failure'
elif [ "$value" == 1 ]; then
echo 'Security alert (Tamper)'
elif [ "$value" == 2 ]; then
echo 'Development mode '
elif [ "$value" == 3 ]; then
echo 'Production mode'
elif [ "$value" == 4 ]; then
echo 'Production-test Mode'
else
echo 'Unexpected value in /sys/kernel/secure/state'
fi

PrintHeader "Device Type"
cat /sys/kernel/secure/type
cat /sys/kernel/secure/subtype

PrintHeader "Serial"
cat /etc/platform/sys/SERIAL

PrintHeader "CPUID"
cat /proc/cpuid


PrintHeader "MCU CPUID"
if [ -e /sys/class/ifboard/sim_sdid ]
then
    cat  /sys/class/ifboard/sim_sdid
fi

PrintHeader "Interface board firmware version"
if [ -e /sys/class/ifboard/fw_version ]
then
    cat /sys/class/ifboard/fw_version
fi

PrintHeader "Terminal Parts"
cat /etc/platform/sys/README


PrintHeader "List of installed packages"
if [[ $IS_ROOT -eq 1 ]]; then
    echo "Platform (rpm -qa):"
    rpm -qa | sort
    echo "Applications:"
    rpm -qa --dbpath /var/lib/installer/database
else
    dbus-send --system  --print-reply --dest=com.equinoxpayments.installer /Server com.equinoxpayments.installer.Server.ListPackages
fi

PrintHeader "Installed SW Packages"
cat /var/lib/installer/swpack/*

PrintHeader "Content of /boot dir"
if [ -e /boot ]
then
    find /boot -type f -exec md5sum {} \;
fi

PrintHeader "Resources"
/home/platform/cgi-bin/resource

PrintHeader "Injected Key IDs"
/home/platform/cgi-bin/key

PrintHeader "Installer logs"
dbus-send --system  --print-reply --dest=com.equinoxpayments.installer /Server com.equinoxpayments.installer.Server.GetStatLog

PrintHeader "Free RAM"
free

PrintHeader "Free flash space"
df -h

PrintHeader "Boot args (/proc/cmdline)"
cat /proc/cmdline

PrintHeader "Network info"
ifconfig
route -n
PrintHeader "Network DNS"
cat /etc/resolv.conf
echo ""
/sbin/resolvconf -l

PrintHeader "Power button state"
cat /sys/kernel/keypad/pwrbtn

PrintHeader "CPU Frequency"
cat /sys/kernel/secure/freq

PrintHeader "Battery status"
if [[ -e /sys/kernel/ifboard/pmic_events/batt_soc ]]; then
    cat /sys/kernel/ifboard/pmic_events/batt_soc
fi

if [[ -e /sys/kernel/ifboard/pmic_events/ch_state ]]; then
    cat /sys/kernel/ifboard/pmic_events/ch_state
fi

if [[ -e /sys/kernel/ifboard/pmic_events/ch_details ]]; then
    cat /sys/kernel/ifboard/pmic_events/ch_details
fi

if [[ -e /sys/class/pmic/batt_age ]]; then
    cat /sys/class/pmic/batt_age
fi

# Coin battery voltage

PrintHeader "Coin battery"
curl http://localhost/cgi-bin/platform/coin-battery-voltage

PrintHeader "LCD type"
cat /sys/kernel/secure/lcdtype

PrintHeader "Running processes (ps)"
ps -T -l -w

PrintHeader "CPU Usage"
top -bn1

PrintHeader "MQX version"
platform_utils -msnk

PrintHeader "Command history (/.ash_history files)"
echo "Superuser (root):"
if [[ -e /.ash_history ]]; then
   ls -latr /.ash_history
    while read -r line || [[ -n "$line" ]]; do
       echo "# $line"
   done < "/.ash_history"
   if [[ $? -ne 0 ]]; then
       echo "Can't retrieve"
   fi
fi;

echo "Superuser (root) 2:"
if [[ -e /root/.ash_history ]]; then
   ls -latr /root/.ash_history
   while read -r line || [[ -n "$line" ]]; do
       echo "# $line"
   done < "/root/.ash_history"
   if [[ $? -ne 0 ]]; then
       echo "Can't retrieve"
   fi
fi;

find /home/ -name .ash_history | while read user; do
   echo "Path: $user:"
   ls -latr "$user"
   while read -r line || [[ -n "$line" ]]; do
       echo "# $line"
   done < "$user"
   if [[ $? -ne 0 ]]; then
       echo "Can't retrieve"
   fi
done;

PrintHeader "Linux version (build version)"
cat /proc/version
cat /etc/*-release

PrintHeader "Syslog messages"
cat /var/log/messages.0
echo "*********************"
cat /var/log/messages

PrintHeader "Dmesg messages"
dmesg

PrintHeader "RPM Database recovery logs"
cat /var/log/rpmdb.log

PrintHeader "RPM verify log"
rpm -V -qa \* ; rpm -V -qa \* --dbpath /var/lib/installer/database/

PrintHeader "Registry settings"
curl localhost/cgi-bin/variable/get

PrintHeader "Controllers"
/home/platform/cgi-bin/certificates

PrintHeader "libc/toolchain version"
/lib/libc-*.so

PrintHeader "File System"
find / \
    -not -path "/dev/block/*" \
    -not -path "/dev/bus/*" \
    -not -path "/dev/char/*" \
    -not -path "/mnt/*" \
    -not -path "/proc/*" \
    -not -path "/run/*" \
    -not -path "/sys/class/dma/*" \
    -not -path "/sys/devices/platform/nbs_rfid_as3911/*" \
    -not -path "/sys/devices/platform/soc/*" \
    -not -path "/sys/devices/virtual/*" \
    -not -path "/sys/firmware/devicetree/*" \
    -not -path "/sys/kernel/slab/*" \
    -exec ls -dlis --full-time {} + 2>/dev/null

PrintHeader "Mounted Devices"
cat /proc/mounts

PrintHeader "USB devices"
lsusb
echo " ** Details ** "
for sysdevpath in $(find /sys/bus/usb/devices/usb*/ -name dev); do
    syspath="${sysdevpath%/dev}"
    devname="$(udevadm info -q name -p $syspath)"
    echo $devname |grep "^bus/*" > /dev/null
    if [[ $? -eq 0 ]]; then  continue; fi
    echo "" && udevadm info -q property --export -p $syspath
done

PrintHeader "Opened File desriptors"
if [[ $IS_ROOT -ne 1 ]]; then
    echo "** Incomplete list **"
fi
lsof

PrintHeader "Checksum"
# Don't try if not a root
if [[ $IS_ROOT -eq 1 ]]; then
    freespace="$( df | grep ubi | awk '{print $4}' )"
    if [ $freespace -lt 20000 ]
    then
        echo "Not enough space on disk to calculate"
    exit 1;
    fi

    cwd=$(pwd)
    LOOPDEV=$(losetup -f)
    cd /root
    rm -rf /root/CHECK &> /dev/null
    mkdir /root/CHECK
    cd /root/CHECK && rm ./MTD1.dump &> /dev/null
    nanddump -f   MTD1.dump /dev/mtd1 &> /dev/null
    sync
    losetup $LOOPDEV  MTD1.dump
    rm -rf ./MNT1 && mkdir ./MNT1 && mount $LOOPDEV ./MNT1
    sync

    mtd1_keyboard_hash="$( md5sum /root/CHECK/MNT1/boot/keyboard | awk '{print $1}' )"
    mtd1_linux_hash="$( md5sum /root/CHECK/MNT1/boot/linux | awk '{print $1}' )"
    mtd1_secure_hash="$( md5sum /root/CHECK/MNT1/boot/secure | awk '{print $1}' )"
    mtd1_sred_hash="$( md5sum /root/CHECK/MNT1/boot/sred | awk '{print $1}' )"

    losetup -d $LOOPDEV
    umount ./MNT1 && rm -rf ./MNT1
    rm ./MTD1.dump &> /dev/null
    rm -rf /root/CHECK/MNT1

    rm ./MTD2.dump &> /dev/null
    nanddump -f   MTD2.dump /dev/mtd2 &> /dev/null
    sync
    losetup $LOOPDEV  MTD2.dump
    rm -rf ./MNT2 && mkdir ./MNT2 && mount $LOOPDEV ./MNT2
    sync

    mtd2_keyboard_hash="$( md5sum /root/CHECK/MNT2/boot/keyboard | awk '{print $1}' )"
    mtd2_linux_hash="$( md5sum /root/CHECK/MNT2/boot/linux | awk '{print $1}' )"
    mtd2_secure_hash="$( md5sum /root/CHECK/MNT2/boot/secure | awk '{print $1}' )"
    mtd2_sred_hash="$( md5sum /root/CHECK/MNT2/boot/sred | awk '{print $1}' )"

    losetup -d $LOOPDEV
    umount ./MNT2 && rm -rf ./MNT2
    rm ./MTD2.dump &> /dev/null
    rm -rf /root/CHECK

    if [ "$mtd1_keyboard_hash" != "$mtd2_keyboard_hash" ]
    then echo "mtd1.keyboard and mtd2.keyboard MD5 do not match: " $mtd1_keyboard_hash " vs " $mtd2_keyboard_hash
    else echo "keyboard MD5: "$mtd1_keyboard_hash;
    fi
    if [ "$mtd1_linux_hash" != "$mtd2_linux_hash" ]
    then echo "mtd1.linux and mtd2.linux MD5 do not match: " $mtd1_linux_hash " vs " $mtd2_linux_hash
    else echo "linux MD5:    "$mtd1_linux_hash;
    fi
    if [ "$mtd1_secure_hash" != "$mtd2_secure_hash" ]
    then echo "mtd1.secure and mtd2.secure MD5 do not match: " $mtd1_secure_hash " vs " $mtd2_secure_hash
    else echo "secure MD5:   "$mtd1_secure_hash;
    fi
    if [ "$mtd1_sred_hash" != "$mtd2_sred_hash" ]
    then echo "mtd1.sred and mtd2.sred MD5 do not match: " $mtd1_sred_hash " vs " $mtd2_sred_hash
    else echo "sred MD5:     "$mtd1_sred_hash;
    fi
    cd $cwd

else
    echo "Can't retrieve"
fi

