#!/bin/bash
#==============================================================================
#  Company:
#      Equinox Payments, LLC
#
#  Product:
#      Application Installer Service
#      Copyright 2011-2019 Equinox Payments, LLC
#
#  \brief
#      Script that restarts services after application installation
#
#==============================================================================

# List of services that should be restarted after application update
SERVICE_TO_RESTART_0="/etc/platform/32-securityservice;/etc/platform/34-arb-printer;/etc/platform/34-sredd;/etc/platform/35-browser"

# List of services that should be restarted after application installation (before result is sent to caller)
SERVICE_TO_RESTART_1="/etc/platform/32-securityservice;/etc/platform/34-arb-printer;/etc/platform/34-sredd;/etc/platform/35-browser"

# List of services that should be restarted after application installation (after result is sent to caller)
SERVICE_TO_RESTART_2="/etc/platform/34-fastcgi;/etc/rc.d/init.d/lighthttpd"

# List of services that should reload configuration after new application is installed
SERVICE_TO_RELOAD_0="/etc/rc.d/init.d/messagebus"

# List of services to stop on low-memory
SERVICE_TO_STOP="/etc/platform/35-browser"

# Restart services passed in $1
RestartServices()
{
    serviceList=`echo ${1} | tr ';' '\n'`
    for service in ${serviceList}; do
        if [ -f "${service}" ]; then
            ${service} stop
            if [ $? -ne 0 ];then
                echo "Failed to stop ${service}"
                return $?
            fi
        else
            echo "No script ${service}"
            return 1
        fi
    done;

    for service in ${serviceList}; do
        if [ -f "${service}" ]; then
            ${service} start
            if [ $? -ne 0 ];then
                echo "Failed to start ${service}"
                return $?
            fi
        else
            echo "No script ${service}"
            return 1
        fi
    done;

}

# Stop services passed in $1
StopServices()
{
    serviceList=`echo ${1} | tr ';' '\n'`
    for service in ${serviceList}; do
        if [ -f "$service" ]; then
            ${service} stop
            if [ $? -ne 0 ];then
                 echo "Failed to stop ${service}"
                 return $?
            fi
        else
            echo "No script ${service}"
            return 1
        fi
     done;
}

# Reload services passed in $0
ReloadServices()
{
    serviceList=`echo ${1} | tr ';' '\n'`
    for service in ${serviceList}; do
        if [ -f "$service" ]; then
            ${service} reload
            if [ $? -ne 0 ];then
                echo "Failed to reload ${service}"
                return $?
            fi
        else
            echo "No script ${service}"
            return 1
        fi
    done;
}

# Check if all services that can be restarted by installer exists.
# This function should be executed during rootfs build (post_build.sh).
SelfCheck()
{
     retval=0
     ROOT_DIR="$( cd "${1}" && pwd )"
     serviceList=`cat $0 | grep "^[ ]*SERVICE_TO_.*=" | cut -d"=" -f2 | tr -d '"' | tr ';' '\n' | sort -u`
     for service in ${serviceList}; do
         if [[ ! -e "$ROOT_DIR""$service" ]] && [[ ! -L "$ROOT_DIR""$service" ]] ; then
             echo "Requested startup-script "$ROOT_DIR"$service doesn't exist"
             retval=1
         fi
     done
     return $retval
}

if [[ "$1" = "restart0" ]]; then
    conf_service_screens
    /home/platform/cgi-bin/certreload
    RestartServices ${SERVICE_TO_RESTART_0}
elif [[ "$1" = "restart1" ]]; then
    conf_service_screens
    /home/platform/cgi-bin/certreload
    RestartServices ${SERVICE_TO_RESTART_1}
elif [[ "$1" = "restart2" ]]; then
    RestartServices ${SERVICE_TO_RESTART_2}
elif [[ "$1" = "reload0" ]]; then
    ReloadServices ${SERVICE_TO_RELOAD_0}
elif [[ "$1" = "selfcheck" ]];then
    SelfCheck "${2}"
elif [[ "$1" = "stop" ]];then
    StopServices ${SERVICE_TO_STOP}
elif [[ "$1" = "reboot" ]]; then 
     conf_service_screens
    /home/platform/cgi-bin/certreload
else
    echo "Wrong action requested"
    false
fi

