/**
 * @module jphoenix
 */

/*!
 * jphoenix JavaScript Library
 * ===========================
 *
 * Version 0.21	2017-04-13
 *
 * This file is part of the Luxe platform software and should be loaded
 * to directory /var/run/www/common/
 *
 * Steve's comments.  All of the functions in this library should have
 * a comments section at the start of the function that follows the Linux
 * man-page style of documentation.
 *
 * Generally, all man pages should follow a standard layout to ensure
 * readability and compatibility.
 *
 *
 * NAME
 *	The name of the function
 *
 * SYNOPSIS
 *	One line synopsis with list of calling parameters
 *
 * DESCRIPTION
 *	A detailed drecription of the purpose and usage of
 *	the function.
 *
 * OPTIONS
 *	Describe the options if applicable.
 *
 * EXAMPLES
 *	Examples are always nice to have.
 *
 * SEE ALSO
 *	A list of related commands or functions.
 *
 *
 * EXIT STATUS / RETURNS
 *	Define the normal and exception return values.
 *
 * ENVIRONMENT
 *	Optional
 *
 * FILES
 *	optional
 *
 * For a description of the end user documentation for the jphoenix.js
 * functions, please see the document
 *	"Luxe SDK Application Programmers Guide" Chapter 4 - jphoenix Library.
 *
 *
 * NOTES: Many of the jphoenix functions were originally written as simple
 * wrappers around the HTTP API calls.  However, as the JavaScript code is
 * executing on the Luxe terminal, rather than send all requests to the
 * low level functions that process the requests through the web server, a
 * faster execution path is simply to call the low level functions directly.
 * Hence, you will see calls to secure.xxx when calling the promptxxx functions
 * For instance, the promptClear() function in this library used to make an
 * HTTP request to http://127.0.0.1/cgi-bin/prompt/clear (which in turn
 * calls secure.clear) but now it directly calls prompt.clear
 */

// ----------------------------settings------------------------------
var checkIPAddress = window.location.host;

/**

 * @description
 * TODO missing description
 *
 * @private
 *
 */
var jPhoenixSettings = {

    /**
     * variables
     */

    HTTP_PROTOCOL: 'http',
    DEVICE_IP: window.location.host,

    /**
     * functions
     */

    getBaseUrl: function () {
        var ip = this.DEVICE_IP;
        if (isTerminal()) {
            ip = 'localhost';
        }
        return jPhoenixSettings.HTTP_PROTOCOL + '://' + ip + '/cgi-bin/';
    }
};

/**

 * @description
 * Steve's note.  This was only really useful while developing jphoenix.
 *
 * As the jphoenix library is a set of functions that are only executed
 * on the Luxe terminals browser, none of the jphoenix functions need to
 * worry about running on a PC's browser any more.
 *
 * On second thoughts, it may be useful to keep and enhance the idea of
 * running the functions within a PC's browser as PC browsers have good
 * web developer tools usually built in to the browser.
 *
 * @return {boolean} returns true if script is executed on terminal, false otherwise
 *
 * @private
 *
 */
function isTerminal() {
    if (typeof nfc === 'undefined'
        || typeof p2pe === 'undefined'
        || typeof scr === 'undefined'
        || typeof page === 'undefined'
        || typeof utils === 'undefined'
        || typeof registry === 'undefined'
        || typeof secure === 'undefined'
        || typeof installer === 'undefined'
        || typeof camera === 'undefined') {
        return false;
    }
    return true;
}

/*
 * ============================================================================================================
 *    				                            logging
 * ============================================================================================================
 */

/**

 *  @description
 *	This object contains function that send log entries with appropriate log level to log file.
 *
 *  logger.error( msg );
 *  logger.warn( msg );
 *  logger.info( msg );
 *  logger.debug( msg );
 *  @param {string} msg - log entry to be send to log file
 *
 *  @example
 *  logger.error('error with statusCode={} occurs in function={}', statusCode, functionName);
 *
 * @instance
 *
 */
var logger = {
    levelOrder: ['debug', 'info', 'warn', 'error', 'off'],
    logLevel: 'off',
    component: 'jphoenix',
    setLogLevel: function (level) {
        this.logLevel = level;
    },
    error: function () {
        logger._toLog(Array.prototype.slice.call(arguments), 'error');
    },
    warn: function () {
        logger._toLog(Array.prototype.slice.call(arguments), 'warn');
    },
    info: function () {
        logger._toLog(Array.prototype.slice.call(arguments), 'info');
    },
    debug: function () {
        logger._toLog(Array.prototype.slice.call(arguments), 'debug');
    },
    _toLog: function (args, level) {
        var levelReached = false;
        for (var idx = 0; idx < this.levelOrder.length; idx++) {
            if (this.levelOrder === 'off') {
                return;
            }
            if (this.levelOrder[idx] === this.logLevel && levelReached) {
                return;
            } else if (this.levelOrder[idx] === this.logLevel) {
                break;
            }
            if (this.levelOrder[idx] === level) {
                levelReached = true;
            }
        }
        var msg = logger._format(args[0], args.slice(1, args.length));
        log[level]("[" + logger.component + "] [" + level.toUpperCase() + "] " + msg);

    },
    _format: function (source, params) {
        var i = 0;
        var output = source.replace(/\{}/g, function () {
            var substArray = params.slice(i, ++i);
            var subst = substArray && substArray.length ? substArray[0] : undefined;
            if (subst instanceof Object) {
                subst = JSON.stringify(subst);
            }
            if (subst && subst.length > 512) {	//limit length
                subst = subst.substr(0, 512) + "...";
            }
            return (subst);
        });
        return (output);
    }
};

/*
 * ============================================================================================================
 *    								   	 jphoenix layer error codes
 * ============================================================================================================
 */

var eqErrors = {
    UNKNOWN_ERROR: {errorCode: 0, errorMessage: "Unknown error"},
    PROMPT_MSR_ERROR: {errorCode: 1, errorMessage: "promptMSR error"},
    PROMPT_CLEAR_ERROR: {errorCode: 2, errorMessage: "promptClear error"},
    PROMPT_TOUCH_AREA_ERROR: {errorCode: 3, errorMessage: "promptTouchArea error"},
    PROMPT_TEXT_ERROR: {errorCode: 4, errorMessage: "promptText error"},
    PROMPT_SIGNED_IMAGE_ERROR: {errorCode: 5, errorMessage: "promptSignedImage error"},
    PROMPT_CHECK_BOX_ERROR: {errorCode: 6, errorMessage: "promptCheckbox error"},
    PROMPT_RADIO_BOX_ERROR: {errorCode: 7, errorMessage: "promptRadiobox error"},
    PROMPT_TEXT_BOX_ERROR: {errorCode: 8, errorMessage: "promptTextbox error"},
    PROMPT_SIGNATURE_ERROR: {errorCode: 9, errorMessage: "promptSignature error"},
    PROMPT_PIN_BLOCK_ERROR: {errorCode: 10, errorMessage: "promptPinBlock error"},
    PROMPT_EDIT_BOX_ERROR: {errorCode: 11, errorMessage: "promptEditBox error"},
    PROMPT_MANUAL_PAN_ERROR: {errorCode: 12, errorMessage: "promptManualPAN error"},
    PROMPT_MANULA_CVV2_ERROR: {errorCode: 13, errorMessage: "promptManualCVV2 error"},
    PROMPT_EXP_DATE_ERROR: {errorCode: 14, errorMessage: "promptExpDate error"},
    PAGE_SHOW_ERROR: {errorCode: 15, errorMessage: "pageShow error"},
    PROMPT_EVENT_ERROR: {errorCode: 16, errorMessage: "promptEvent error"},
    SET_CONTROL_VALUE_ERROR: {errorCode: 17, errorMessage: "setControlValue error"},

    MSR_TRACK_NOT_FOUND: {errorCode: 100, errorMessage: "MSR track not received."},
    MSR_GENERAL_ERROR: {errorCode: 199, errorMessage: "MSR general error."},
    MANUALPANCVV2_GENERAL_ERROR: {errorCode: 299, errorMessage: "Manual PAN / CVV general error."},
    EXPIRATION_DATE_INVALID: {errorCode: 300, errorMessage: "Expiration date format is not valid - should be in mm/yy format (mm - number 1-12, yy - number 00-99)"},
    EXPIRATION_DATE_UNABLE_TO_SET: {errorCode: 301, errorMessage: "Unable to set expiration date"},
    CARD_INSERTED_ERROR: {errorCode: 400, errorMessage: "ATR failed, please remove card and try again"},
    NFC_ON_UI_REQUEST_DATA_ERROR: {errorCode: 500, errorMessage: "nfc.onUIRequest Invalid data"},
    NFC_ON_PLUGIN_PROCESSED_GENERAL: {errorCode: 501, errorMessage: "nfc.onPluginProcessed Invalid data"},
    NFC_ON_PLUGIN_PROCESSED_ANDROID: {errorCode: 502, errorMessage: "nfc.onPluginProcessed AndroidPay Error"},
    NFC_ON_PLUGIN_PROCESSED_APPLE: {errorCode: 503, errorMessage: "nfc.onPluginProcessed ApplePay Error"},

    ACTION_BEEP_ERROR: {errorCode: 600, errorMessage: "Beep error"},
    ACTION_SET_SPEAKER_VOLUME_ERROR: {errorCode: 601, errorMessage: "Unable to set speaker volume"},
    ACTION_GET_SPEAKER_VOLUME_ERROR: {errorCode: 602, errorMessage: "Unable to get speaker volume"},
    ACTION_RUN_PROCESS_ERROR: {errorCode: 603, errorMessage: "runProcess error"},
    ACTION_ENABLE_CONTROL_ERROR: {errorCode: 604, errorMessage: "enableControl error"},
    ACTION_CLEAR_SIGNATURE_ERROR: {errorCode: 605, errorMessage: "clearSignature error"}
};

/*
 * ============================================================================================================
 *    								   	 jphoenix prompt functions
 * ============================================================================================================
 */


var msrTracksToRead, msrMinTracksReadForValidity;
/**
  @description
 *  This function prompts MSR track data.
 *
 *  @param {array} tracksToRead
 *  @param {array} minTracksReadForValidity
 *
 *  @example
 *  promptMSR([1,2,3], [1]);
 *
 * @instance
 *
 */
function promptMSR(tracksToRead, minTracksReadForValidity) {
    logger.debug('promptMSR: tracksToRead={}, minTracksReadForValidity={}', tracksToRead, minTracksReadForValidity);
    msrTracksToRead = tracksToRead;
    msrMinTracksReadForValidity = minTracksReadForValidity;
    var statusCode = secure.promptMSR();
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_MSR_ERROR, {statusCode: statusCode});
    }
}
/**

 *  @description
 *  This function removes (deactivates) all previously enabled touch input areas.
 * 	The touch screen and numeric keypad are disabled.
 *
 *  @param {function} [callBackHandler] - callback function
 *
 * @instance
 *
 */
function promptClear(callBackHandler) {
    //
    // For now, just set aync to false to ensure that the prompt clear
    // is completed before sending the buttons.
    // I should update this to look at the response from clear
    //		if(checkIPAddress==="localhost" && checkIPAddress!=undefined){
    logger.debug('promptClear');
    var statusCode = secure.promptClear();
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_CLEAR_ERROR, {statusCode: statusCode});
    }
    if (callBackHandler) {
        callBackHandler(statusCode);
    }
}

/**

 *  @description
 *  This function enables a touch input area on the screen.
 *
 *  @param {string} name - name of the touch area
 *  @param {string} onTouchImg - resourceName of the image on touch area touched  is optional
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} width - width size of the touch area
 *  @param {number} height - height of the touch area
 *  @param {string} returnText - text to be return to caller when touch area is touched
 *
 *  @example
 *	promptTouchArea( 'userResponse','', 10, 390, 180, 77, 'ok');
 *
 * @instance
 *
 */
function promptTouchArea(name, onTouchImg, x, y, width, height, returnText) {
    if (onTouchImg && onTouchImg.charAt(0) === '$') {
        onTouchImg = variableGet(onTouchImg.substring(1));
    }
    logger.debug('promptTouchArea: name={}, onTouchImg={}, x={}, y={}, width={}, height={}, returnText={}', name, onTouchImg, x, y, width, height, returnText);
    var statusCode = secure.promptButton(name, returnText, x, y, width, height, onTouchImg);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_TOUCH_AREA_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *  This function is used to add a button image with
 * 	dynamic button label text and enable a touch input area over a screen background image
 *
 *	This function is a combination of the text, image, and button controls with some
 *	JavaScript thrown in to control the display of the image and touch input based on
 *	the value of the text string.
 *
 *	If the label is evaluated to be empty (NULL), the image is NOT displayed
 *	and the touch input area is not enabled.  This allows for the generation of run time
 *	buttons such an an EMV card specific language selection button or application selection
 *	where the number of matching languages / applications is determined at run time after
 *	a smart card is inserted.
 *
 * 	@param {string} appName - application name
 * 	@param {string} langCode - language code is optional - default is defaultLanguageCode form prompts.json
 * 	@param {string} domId - ID of the "div" tag to update with the button image and the button text
 * 	@param {string} name - name of the button. This need not be unique
 * 	@param {string} label - text label to show on the button. If empty, the button won't be shown
 * 	@param {string} returnText - text string to return when the defined button is touched
 *	@param {number} x - x co-ordinate
 *	@param {number} y - y co-ordinate
 * 	@param {number} width - width size of the button area
 * 	@param {number} height - height of the button area
 * 	@param {string} [onTouchImg] - name of a previously loaded secure resource image that will be shown while the area is touched.
 * 	@param {boolean} langControlled - whether the image is language controlled
 *
 *  @example
 *	promptLabelButton( 'viking','en', 'ctrl_2', 'lblBtn', 'return text', 10, 390, 180, 77, 'viking');
 *
 * @instance
 *
 */
function promptLabelButton(appName, langCode, domId, name, label, returnText, x, y, width, height, onTouchImg, langControlled, textUpdateDynamically) {
    logger.debug('promptLabelButton: appName={}, domId={}, name={}, label={}, returnText={}, x={}, y={}, width={}, height={}, onTouchImg={}, langCode={}, langControlled={}', appName, domId, name, label, returnText, x, y, width, height, onTouchImg, langCode, langControlled);
    if(langControlled && langCode) {
        setupImage(domId, langCode, langControlled);
    }
    promptText(appName, langCode, domId, label, textUpdateDynamically);
    promptTouchArea(name, onTouchImg, x, y, width, height, returnText);
}

/**

 *  @description
 *  This function parses the text string and displays the completed string at the
 *	DOM with the ID dom_id.
 *
 *	The text string may include one or more of the following elements in any sequence:
 *	A specific string included in single quotes 'Hello'
 *	A variable value $variableName
 *	An index into the prompts.xml file #welcome
 *
 *	The complete string must be included in double quotes and may contain multiple
 *	text strings, multiple prompts.xml indexes and multiple variables.
 *
 *  @param {string} appName - application name
 *  @param {string} langCode - language code is optional - default is defaultLanguageCode form prompts.json
 *  @param {string} domId - ID of the "div" tag to update with the button image and the button text
 *  @param {string} varText - text that needs to be translated
 *
 *  @example
 *  "#greeting 'Steve' #transAmount $amount"
 *
 *  If the current terminal language is English, and the prompts.xml English index
 *  #greeting is 'Welcome' and the English prompts.json index #transAmount is
 *  'Transaction Amount' and the variable $amount has the value '$12.34' then the completed
 *  string will be
 *
 *  "WelcomeSteveTransaction Amount$12.34".
 *
 *  Obviously we need to include some spaces such as
 *  "#greeting ' Steve, ' #transAmount ' ' $amount" and this should display as:
 *
 *  "Welcome Steve, Transaction Amount $12.34"
 *
 * @instance
 *
 */
function promptText(appName, langCode, domId, varText, textUpdateDynamically) {
    logger.debug('promptText: appName={}, langCode={}, domId={}, varText={}, textUpdateDynamically={}', appName, langCode, domId, varText, textUpdateDynamically);
    var translation = parsePlaceholderString(langCode, varText);
    if (translation.text) {
        document.getElementById(domId).querySelector('.label').innerHTML = translation.text;
    }
    translation.variables.forEach(function (varName) {
        registerForVariable(varName, function () {
            if(textUpdateDynamically || typeof textUpdateDynamically === "undefined"){
                translation = parsePlaceholderString(langCode, varText);
                document.getElementById(domId).querySelector('.label').innerHTML = translation.text || '';
            }
        });
    });
}

/**

 *  @description
 *	This function tells MQX to display the secure resource of type = Background onto the
 *	display.  This also sets up an area where there may be >8 touch input areas defined.
 *	This is required if there is a numeric or alphanumeric keyboard to be displayed on
 *	the LCD as there will be >= 10 touch areas required.
 *
 *  @param {string} resourceName - name of a currently installed secure Background resource
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {string} langCode - language code
 *  @param {boolean} langControlled - whether the image is language controlled
 *
 *  @example
 *  promptSignedImage( 'sign', 0, 0 );
 *
 * @instance
 *
 */
function promptSignedImage(resourceName, x, y, langCode, langControlled) {
    if (resourceName && resourceName.charAt(0) === '$') {
        resourceName = variableGet(resourceName.substring(1));
    }
    if (langControlled && langCode) {
        resourceName = updateLangControlledSignedImageName(resourceName, langCode);
    }
    logger.debug('promptSignedImage: resourceName={}, x={}, y={}, langCode={} langControlled={}', resourceName, x, y, langCode, langControlled);
    var statusCode = secure.promptImage(resourceName, x, y);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_SIGNED_IMAGE_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function defines a checkbox input area.
 *	The image on the display is assumed to include the checkbox picture (unchecked).
 *	This allows the user to define exactly what the checkbox looks like.
 *
 *  @param {string} appName - application name
 *  @param {string} langCode - language code is optional - default is defaultLanguageCode form prompts.json
 *  @param {string} domId - ID of the "div" tag to update with the translated text
 *  @param {string} name - name of the checkbox
 *  @param {string} resourceName - resource name of the image on checkbox checked is optional
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} width - width size of the checkbox area
 *  @param {number} height - height of the checkbox area
 *  @param {string} returnTextOff - text that is return when checkbox is unchecked
 *  @param {string} returnTextOn - text that is return when checkbox is checked
 *  @param {number} initialChecked - ( 0 or 1 ) value determines if checkbox is to be checked or not
 *  @param {string} varText - text that needs to be translated
 *
 *  @example
 *	promptCheckbox('checkbox', 'checkbox2', 33, 20, 100, 0, 'select1', 'unselect1', 1);
 *
 * @instance
 *
 */
function promptCheckbox(appName, langCode, domId, name, resourceName, x, y, width, height, returnTextOff, returnTextOn, initialChecked, varText, textUpdateDynamically) {
    if (resourceName && resourceName.charAt(0) === '$') {
        resourceName = variableGet(resourceName.substring(1));
    }
    initialChecked = initialChecked === "true" ? true : initialChecked == "false" ? false : initialChecked;
    if (initialChecked == true) {
        initialChecked = 1;
    } else {
        initialChecked = 0;
    }
    logger.debug('promptCheckbox: appName={}, langCode={}, domId={}, name={}, resourceName={}, x={}, y={}, width={}, height={}, returnTextOff={}, returnTextOn={}, initialChecked={}, varText={}', appName, langCode, domId, name, resourceName, x, y, width, height, returnTextOff, returnTextOn, initialChecked, varText);
    promptText(appName, langCode, domId, varText, textUpdateDynamically);
    var statusCode = secure.promptCheckbox(name, resourceName, returnTextOff, returnTextOn, x, y, width, height, initialChecked);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_CHECK_BOX_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function defines a radiobox input area.  The image on the display is assumed to include the original radiobox picture.
 *	This allows the user to define exactly what the radiobox looks like.
 *
 *  @param {string} appName - application name
 *  @param {string} langCode - language code is optional - default is defaultLanguageCode form prompts.json
 *  @param {string} domId - ID of the "div" tag to update with the translated text
 *  @param {string} name - name of the radiobox
 *  @param {string} resourceName - resource name of the image on radiobox selected is optional
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} width - width size of the radiobox area
 *  @param {number} height - height of the radiobox area
 *  @param {string} returnText - text that is return when radiobox is selected
 *  @param {number} initialChecked - ( 0 or 1 ) value determines if radiobox is to be selected or not
 *  @param {string} varText - text that needs to be translated
 *
 *  @example
 *	promptRadiobox('radio', 'radiobox2', 33, 70, 100, 0, 'radio1', 1);
 *
 * @instance
 *
 */
function promptRadiobox(appName, langCode, domId, name, resourceName, x, y, width, height, returnText, initialChecked, varText, textUpdateDynamically) {
    if (resourceName && resourceName.charAt(0) === '$') {
        resourceName = variableGet(resourceName.substring(1));
    }
    initialChecked = initialChecked === "true" ? true : initialChecked == "false" ? false : initialChecked;
    if (initialChecked == true) {
        initialChecked = 1;
    } else {
        initialChecked = 0;
    }
    logger.debug('promptRadiobox: appName={}, langCode={}, domId={}, name={}, resourceName={}, x={}, y={}, width={}, height={}, returnText={}, initialChecked={}, varText={}', appName, langCode, domId, name, resourceName, x, y, width, height, returnText, initialChecked, varText);
    promptText(appName, langCode, domId, varText, textUpdateDynamically);
    var statusCode = secure.promptRadio(name, resourceName, returnText, x, y, width, height, initialChecked);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_RADIO_BOX_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This Function defines a textbox input area.  The image on the display is assumed to include the original textbox picture.
 *	This allows the user to define exactly what the textbox looks like.
 *
 *  @param {string} name - name of the textbox
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} w - width
 *  @param {number} h - height
 *  @param {string} initialString - text shown when textbox is displayed.
 *  @param {string} textTemplate - template that determines the format of the input field
 *  @param {number} valign -  vertical alignment ( CENTRAL_ALIGNMENT = 0, TOP_ALIGNMENT = 1, BOTTOM_ALIGNMENT = 2 )
 *  @param {number} halign -  horizontal alignment ( CENTRAL_ALIGNMENT = 0, LEFT_ALIGNMENT = 1, RIGHT_ALIGNMENT = 2 )
 *  @param {string} kbdname -  keyboard to be displayed for textbox (previosly loaded resource image file)
 *  @param {string} kbdViewName	- keyboard view name to display
 *  @param {number} kbdx -  x co-ordinate for keyboard
 *  @param {number} kbdy -  y co-ordinate for keyboard
 *  @param {string} fontname -  textbox font resource name ( previosly loaded font resouce ), default font is used when empty
 *  @param {string} fgColor - rgb color
 *  @param {string} bgColor - rgb color
 *
 *  @example
 *	promptTextbox("textbox", 155, 105, 460, 40, "", "@@@@@@@@@@", 0, 0, "keyboard0", 0, 200, "TimesRoman12", "rgb(241,229,229)", "rgb(179,52,52)", "kbdViewName");
 *
 * @instance
 *
 */
function promptTextbox(name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdx, kbdy, fontname, fgColor, bgColor, kbdViewName) {
    logger.debug('promptTextbox: name={}, x={}, y={}, width={}, height={}, initialString={}, textTemplate={}, valign={}, halign={}, kbdname={}, kbdViewName={}, kbdx={}, kbdy={}, fontname={}, fgColor={}, bgColor={}', name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdViewName, kbdx, kbdy, fontname, fgColor, bgColor);
    var intBgColor = rgbStringToInt(bgColor);
    var intFgColor = rgbStringToInt(fgColor);
    secure.setBkColor(intBgColor);
    secure.setFrColor(intFgColor);

    var newfont = (null == fontname) ? "" : fontname;
    var statusCode = secure.promptTextbox({
        name: name,
        textTemplate: textTemplate,
        initialString: initialString,
        x: x,
        y: y,
        w: w,
        h: h,
        valign: valign,
        halign: halign,
        kbdx: kbdx,
        kbdy: kbdy,
        kbdname: kbdname,
        kbdViewName: kbdViewName,
        fontname: newfont
    });
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_TEXT_BOX_ERROR, {statusCode: statusCode});
    }
}

var signatureParams;
/**

 *  @description
 *	This function defines an input area that will accept a signature.
 *	The display is assumed to already include the image of a box defining the signature input space.
 *  This allows the user to define exactly what the signature input area looks like.
 *  Signature data is BESE64 encoded. 
 *
 *  @param {string} name - name of the signature box
 *  @param {number} x - x co-ordinate of the signature box
 *  @param {number} y - y co-ordinate of the signature box
 *  @param {number} width - width size of the signature area
 *  @param {number} height- height of the signature area
 *  @param {string} color - color in "rgb(rr,gg,bb)" format
 *  @param {boolean} useLegacyFormat -
 *  @param {number} maxSignaturePoints -
 *
 *  @example
 *	promptSignature( 'signature',  72, 72, 647, 326, "rgb(255, 62, 109)", true, 6000 );
 *
 * @instance
 *
 */
function promptSignature(name, x, y, width, height, rgbColor, useLegacyFormat, maxSignaturePoints) {
    logger.debug('promptSignature: name={}, x={}, y={}, width={}, height={}, rgbColor={}, useLegacyFormat={}, maxSignaturePoints={}', name, x, y, width, height, rgbColor, useLegacyFormat, maxSignaturePoints);
    signatureParams = {name: name, x: x, y: y, width: width, height: height, rgbColor: rgbColor};
    var intColor = 128; //default is blue pen
    if(rgbColor) {
        intColor = rgbStringToInt(rgbColor);
    }
    var statusCode = secure.promptSignature(name, x, y, width, height, intColor, maxSignaturePoints, useLegacyFormat);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_SIGNATURE_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function defines an input area that will accept a signature.
 *	The display is assumed to already include the image of a box defining the signature input space.
 *  This allows the user to define exactly what the signature input area looks like.
 *
 *  @param {string} name - name of the signature box
 *  @param {number} x - x co-ordinate of the signature box
 *  @param {number} y - y co-ordinate of the signature box
 *  @param {number} width - width size of the signature area
 *  @param {number} height- height of the signature area
 *  @param {string} color - color in "rgb(rr,gg,bb)" format
 *  @param {number} format - LEGACY = 0 (BASE64 encoded), ENHANCED = 1 (BASE64 encoded), BMP_MONOCHROME1_HEADER_AND_DATA = 2 (HEX string), BMP_MONOCHROME1_DATA_ONLY = 3 (HEX string), ASCII_3B = 4 (HEX string)
 *  @param {number} maxSignaturePoints -
 *
 *  @example
 *	promptSignature( 'signature',  72, 72, 647, 326, "rgb(255, 62, 109)", 0, 6000 );
 *
 * @instance
 *
 */
function promptSignature2(name, x, y, width, height, rgbColor, format, maxSignaturePoints) {
    logger.debug('promptSignature: name={}, x={}, y={}, width={}, height={}, rgbColor={}, format={}, maxSignaturePoints={}', name, x, y, width, height, rgbColor, format, maxSignaturePoints);
    signatureParams = {name: name, x: x, y: y, width: width, height: height, rgbColor: rgbColor};
    var intColor = 128; //default is blue pen
    if(rgbColor) {
        intColor = rgbStringToInt(rgbColor);
    }
    var statusCode = secure.promptSignature2(name, x, y, width, height, intColor, maxSignaturePoints, format);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_SIGNATURE_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function prompts for, and returns an encrypted PIN block.
 *	The function assumes that the display already contains an image prompting the user to enter their PIN.
 *	This image need not be signed as when the running application invokes a page with a Pinblock control,
 *	the Pinblock function wil only return an encrypted Pinblock and not the user's PIN.
 *
 *  @param {number} timeoutInSec - pin entry screen timeout in seconds
 *  @param {number} interTimeoutInSec - pin entry screen timeout in seconds (between key presses)
 *  @param {number} pinY - Y coordinate of pin entry box
 *  @param {string} fgColor - rgb color
 *  @param {string} bgColor - rgb color
 *  @param {string} keySlotIdSysVarName - system variable name, key slot id value is number
 *  @param {string} masterKeyIdSysVarName - system variable name, master key id value is number
 *  @param {string} keySysVarName - system variable name, master key encrypted session key (contains base64 encoded binary data)
 *  @param {string} verifyOnlineSysVarName - system variable name, value true - online PIN verification, value false - offline PIN verification
 *  @param {string} allowPinEntryBypassSysVarName - system variable name, setting this to false disables *bypass* button during pin entry
 *  @param {string} cardType - could be "EMV card" or "WIC card"
 *
 *  @example
 *  for these system variables
 * $keySlotId =
 * $masterKeyId =
 * $key = 2Fo1cS1kRl/YWjVxLWRGX9haNXEtZEZf
 * $verifyOnline =
 * $allowPinEntryBypass =
 *
 * promptPinBlock( 60, 30, 128, "rgb(241,229,229)", "rgb(179,52,52)", "$keySlotId", "$masterKeyId", "$key", "$verifyOnline", "$allowPinEntryBypass" );
 *
 * @instance
 *
 */
function promptPinBlock(timeoutInSec, interTimeoutInSec, pinY, fgColor, bgColor, keySlotIdSysVarName, masterKeyIdSysVarName, keySysVarName, verifyOnlineSysVarName, allowPinEntryBypassSysVarName, cardType, fontName) {
    logger.debug('promptPinBlock: timeoutInSec={}, interTimeoutInSec={}, pinY={}, fgColor={}, bgColor={}, keySlotIdSysVarName={}, masterKeyIdSysVarName={}, keySysVarName={}, verifyOnlineSysVarName={}, allowPinEntryBypassSysVarName={}, cardType={}, fontName={}',
        timeoutInSec, interTimeoutInSec, pinY, fgColor, bgColor, keySlotIdSysVarName, masterKeyIdSysVarName, keySysVarName, verifyOnlineSysVarName, allowPinEntryBypassSysVarName, cardType, fontName);
    var intBgColor = rgbStringToInt(bgColor);
    var intFgColor = rgbStringToInt(fgColor);
    secure.setBkColor(intBgColor);
    secure.setFrColor(intFgColor);

    var keySlotId, masterKeyId, verifyOnline, key, allowPinEntryBypass;
    if (cardType === 'EMV card') {
        keySlotId = variableGet(keySlotIdSysVarName);
        masterKeyId = variableGet(masterKeyIdSysVarName);
        verifyOnline = variableGet(verifyOnlineSysVarName) === "true";
        key = variableGet(keySysVarName);
        allowPinEntryBypass = variableGet(allowPinEntryBypassSysVarName) === "true";
    } else {
        keySlotId = "0";
        masterKeyId = "0";
        verifyOnline = false;
        key = "";
        allowPinEntryBypass = false;
    }
    logger.debug('promptPinBlock: timeoutInSec={}, interTimeoutInSec={}, pinY={}, fgColor={}, bgColor={}, keySlotId={}, masterKeyId={}, key={}, verifyOnline={}, allowPinEntryBypass={}, fontName={}',
        timeoutInSec, interTimeoutInSec, pinY, fgColor, bgColor, keySlotId, masterKeyId, key, verifyOnline, allowPinEntryBypass, fontName);
    /*
     // key example (in ASCIIHEX): D85A35712D64465FD85A35712D64465FD85A35712D64465F
     var keyArray = [];
     if(key) {
     var hexArray = key.match(/(..?)/g);
     for (var idx in hexArray) {
     var hex = hexArray[idx];
     keyArray.push(parseInt(hex, 16));
     //                logger.debug("i={}, byte={}", idx, parseInt(hex, 16));
     }
     }
     */
    //key example (in BASE64): 2Fo1cS1kRl/YWjVxLWRGX9haNXEtZEZf
    var keyArray = [];
    if (key) {
        var keyEncoded = atob(key);
        for (var idx in keyEncoded) {
            keyArray.push(keyEncoded.charCodeAt(idx));
        }
    }

    logger.debug('promptPinBlock: secure.requestPIN(): keySlotId={}, masterKeyId={}, timeoutInSec={}, interTimeoutInSec={}, pinY={}, keyArray={}, verifyOnline={}, allowPinEntryBypass={}, fontName={}',
        keySlotId, masterKeyId, timeoutInSec, interTimeoutInSec, pinY, keyArray, verifyOnline, allowPinEntryBypass, fontName);

    var statusCode = secure.requestPIN(keySlotId, masterKeyId, timeoutInSec, interTimeoutInSec, pinY, keyArray, verifyOnline, allowPinEntryBypass, fontName);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_PIN_BLOCK_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function prompts for Edit Box with some default values if defined or else it accepts number values
 *	for amounts with a decimal point.
 *
 *  @param {number } promptResourceId
 *  @param {number } promptId
 *  @param {number } titleResourceId
 *  @param {number } titleId
 *  @param {string } text
 *  @param {number } timeout - optional
 *
 *  @example
 *  promptEditBox( 160, 1, 170, 1, 30, "888" );
 *
 * @instance
 *
 */
function promptEditBox(promptResourceId, promptId, titleResourceId, titleId, timeout, text) {
    logger.debug('promptEditBox: promptResourceId={}, promptId={}, titleResourceId={}, titleId={}, timeout={}, text={}', promptResourceId, promptId, titleResourceId, titleId, timeout, text);
    var statusCode = secure.promptEditbox(promptResourceId, promptId, titleResourceId, titleId, timeout, text);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_EDIT_BOX_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *  This function enables a scrollable area on the screen.
 *
 *  @param {string} id - scrollable area id
 *  @param {number} width - width of the scrollable area
 *  @param {number} height - height of the scrollable area
 *  @param {string} areaType - could be table, buttons or text
 *  @param {number} rowsPerTouch - number of lines for single up/down arrow touch
 *
 *  @example
 *  setupScrollableArea( 'cretl_3', 10, 390, 180, 77, 'table');
 *
 * @instance
 *
 */
function setupScrollableArea(id, width, height, areaType, rowsPerTouch) {
    logger.debug('setupScrollableArea: id={}, width={}, height={}, areaType={}, rowsPerTouch={}', id, width, height, areaType, rowsPerTouch);
    var $scrollUp = document.querySelector('#' + id + ' .equ-scroll-up');
    var $scrollDown = document.querySelector('#' + id + ' .equ-scroll-down');

    if (areaType === 'table') {
        var $table = document.querySelector('#' + id + '.equ-scrollable-area table');
        $table.querySelector('tbody').style.height = (height - $table.offsetHeight) + 'px';
    } else if (areaType === 'buttons') {
        var equButtons = document.querySelectorAll('#' + id + ' .equ-button');
        for (var i = 0; i < equButtons.length; i++) {
            var btn = equButtons[i];
            var btnId = id + '-btn-' + i;
            var btnRect = btn.getBoundingClientRect();
            promptTouchArea(btnId, '', btnRect.left, btnRect.top, btnRect.width, btnRect.height, 'scrollablebtn');
        }
    }

    var scrollUpId = id + '-up';
    var scrollDownId = id + '-down';
    var scrollUpRect = $scrollUp.getBoundingClientRect();
    var scrollDownRect = $scrollDown.getBoundingClientRect();
    promptTouchArea(scrollUpId, '', scrollUpRect.left, scrollUpRect.top, scrollUpRect.width, $scrollUp.height, 'scrollup');
    promptTouchArea(scrollDownId, '', scrollDownRect.left, scrollDownRect.top, scrollDownRect.width, scrollDownRect.height, 'scrolldown');

    enableControl(scrollUpId, false);
    enableControl(scrollDownId, false);

    registerForEvent("onTouched", scrollUpId, function (name, text, response) {
        scrollableArea.up(id, areaType, rowsPerTouch);
        executeCallBackFunction("onScrollUp", id, response);
    });
    registerForEvent("onTouched", scrollDownId, function (name, text, response) {
        scrollableArea.down(id, areaType, rowsPerTouch);
        executeCallBackFunction("onScrollDown", id, response);
    });
}

var _activeManualPrompt;
/**

 *  @description
 *	This Function defines a manual pan area.
 *
 *  @param {string} name - name of the manualPAN
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} w - width
 *  @param {number} h - height
 *  @param {string} initialString - text shown when pan is displayed.
 *  @param {string} textTemplate - template that determines the format of the input field
 *  @param {number} valign -  vertical alignment (CENTRAL_ALIGNMENT = 0, TOP_ALIGNMENT = 1, BOTTOM_ALIGNMENT = 2, CURRENT_ALIGNMENT = 3)
 *  @param {number} halign -  horizontal alignment (CENTRAL_ALIGNMENT = 0, LEFT_ALIGNMENT = 1, RIGHT_ALIGNMENT = 2, CURRENT_ALIGNMENT = 3)
 *  @param {string} kbdname -  keyboard to be displayed (previosly loaded resource image file)
 *  @param {string} kbdViewName	- keyboard view name to display
 *  @param {number} kbdx -  x co-ordinate for keyboard
 *  @param {number} kbdy -  y co-ordinate for keyboard
 *  @param {string} fontname -  font resource name (previosly loaded font resouce), default font is used when empty
 *
 *  @example
 *  promptManualPAN(155, 105, 460, 40, "", "@@@@@@@@@@", 0, 0, "keyboard0", 0 ,200, "TimesRoman12", 3, 3, "kbdViewName");
 *
 * @instance
 *
 */
function promptManualPAN(name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdx, kbdy, fontname, kbdViewName) {
    logger.debug('promptManualPAN: name={}, x={}, y={}, width={}, height={}, initialString={}, textTemplate={}, valign={}, halign={}, kbdname={}, kbdViewName={}, kbdx={}, kbdy={}, fontname={}', name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdViewName, kbdx, kbdy, fontname);
    _activeManualPrompt = 'manualPAN';
    var newfont = (null == fontname) ? "" : fontname;
    var statusCode = secure.promptManualPAN({
        name: name, textTemplate: textTemplate, initialString: initialString, x: x, y: y, w: w, h: h,
        valign: valign, halign: halign, kbdx: kbdx, kbdy: kbdy, kbdname: kbdname, kbdViewName: kbdViewName, fontname: newfont
    });
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_MANUAL_PAN_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This Function defines a manual pan area.
 *
 *  @param {string} name - name of the manualCVV2
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} w - width
 *  @param {number} h - height
 *  @param {string} initialString - text shown when cvv2 is displayed.
 *  @param {string} textTemplate - template that determines the format of the input field
 *  @param {number} valign -  vertical alignment (CENTRAL_ALIGNMENT = 0, TOP_ALIGNMENT = 1, BOTTOM_ALIGNMENT = 2, CURRENT_ALIGNMENT = 3)
 *  @param {number} halign -  horizontal alignment (CENTRAL_ALIGNMENT = 0, LEFT_ALIGNMENT = 1, RIGHT_ALIGNMENT = 2, CURRENT_ALIGNMENT = 3)
 *  @param {string} kbdname -  keyboard to be displayed (previosly loaded resouce image file)
 *  @param {string} kbdViewName	- keyboard view name to display
 *  @param {number} kbdx -  x co-ordinate for keyboard
 *  @param {number} kbdy -  y co-ordinate for keyboard
 *  @param {string} fontname - font resource name (previosly loaded font resouce), default font is used when empty
 *
 *  @example
 *  promptManualCVV2(155, 105, 460, 40, "", "@@@@@@@@@@", 0, 0, "keyboard0", 0, 200, "TimesRoman12", 3, 3, "kbdViewName");
 *
 * @instance
 *
 */
function promptManualCVV2(name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdx, kbdy, fontname, kbdViewName) {
    logger.debug('promptManualCVV2: name={}, x={}, y={}, width={}, height={}, initialString={}, textTemplate={}, valign={}, halign={}, kbdname={}, kbdViewName={}, kbdx={}, kbdy={}, fontname={}', name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdViewName, kbdx, kbdy, fontname);
    _activeManualPrompt = 'manualCVV2';
    var newfont = (null == fontname) ? "" : fontname;
    var statusCode = secure.promptManualCVV2({
        name: name, textTemplate: textTemplate, initialString: initialString, x: x, y: y, w: w, h: h,
        valign: valign, halign: halign, kbdx: kbdx, kbdy: kbdy, kbdname: kbdname, kbdViewName: kbdViewName, fontname: newfont
    });
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_MANULA_CVV2_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This Function defines a expiration date input area.
 *
 *  @param {string} name - name of the expDate
 *  @param {number} x - x co-ordinate
 *  @param {number} y - y co-ordinate
 *  @param {number} w - width
 *  @param {number} h - height
 *  @param {string} initialString - text shown when expDate is displayed.
 *  @param {number} valign -  vertical alignment (CENTRAL_ALIGNMENT = 0, TOP_ALIGNMENT = 1, BOTTOM_ALIGNMENT = 2)
 *  @param {number} halign -  horizontal alignment (CENTRAL_ALIGNMENT = 0, LEFT_ALIGNMENT = 1, RIGHT_ALIGNMENT = 2)
 *  @param {string} kbdname -  keyboard to be displayed for textbox (previosly loaded resource image file)
 *  @param {string} kbdViewName	- keyboard view name to display
 *  @param {number} kbdx -  x co-ordinate for keyboard
 *  @param {number} kbdy -  y co-ordinate for keyboard
 *  @param {string} fontname -  expDate font resource name (previosly loaded font resouce), default font is used when empty
 *  @param {string} fgColor - rgb color
 *  @param {string} bgColor - rgb color
 *
 *  @example
 *  promptExpDate("textbox", 155, 105, 460, 40, "", 0, 0, "keyboard0", 0, 200, "TimesRoman12", "rgb(241,229,229)", "rgb(179,52,52)", "kbdViewName");
 *
 * @instance
 *
 */
function promptExpDate(name, x, y, w, h, initialString, valign, halign, kbdname, kbdx, kbdy, fontname, fgColor, bgColor, kbdViewName) {
    var textTemplate = 'mm/yy';
    logger.debug('promptExpDate: name={}, x={}, y={}, width={}, height={}, initialString={}, textTemplate={}, valign={}, halign={}, kbdname={}, kbdViewName={}, kbdx={}, kbdy={}, fontname={}, fgColor={}, bgColor={}', name, x, y, w, h, initialString, textTemplate, valign, halign, kbdname, kbdViewName, kbdx, kbdy, fontname, fgColor, bgColor);
    var intBgColor = rgbStringToInt(bgColor);
    var intFgColor = rgbStringToInt(fgColor);
    secure.setBkColor(intBgColor);
    secure.setFrColor(intFgColor);

    var newfont = (null == fontname) ? "" : fontname;
    var statusCode = secure.promptTextbox({
        name: name, textTemplate: textTemplate, initialString: initialString, x: x, y: y, w: w, h: h,
        valign: valign, halign: halign, kbdx: kbdx, kbdy: kbdy, kbdname: kbdname, kbdViewName: kbdViewName, fontname: newfont
    });
    if (statusCode != 0) {
        throw new EQError(eqErrors.PROMPT_EXP_DATE_ERROR, {statusCode: statusCode});
    }
}

var contactlessLEDControl;
/**

 *  @description
 *	This Function defines a led panel.
 *
 *  @param {string} id - ID of the <div> tag to update with the led state
 *  @param {string} type - type of led panel (Generated or Custom)
 *  @param {number} width - single led width
 *  @param {number} height - single led height
 *  @param {number} spacing - single led margin
 *  @param {string} imageOn - custom led on
 *  @param {string} imageOff - custom led off
 *  @param {string} colorOn - rgb color for generated led on
 *  @param {string} colorOff - rgb color for generated led off
 *  @param {string} radius - generated led radius
 *
 *  @example
 *  promptContactlessLED("ctrl_15", "Custom", 50, 50, 20, "/html/<appName>/assets/images/20170503_153629.jpg", "/html/global/assets/images/viking.png", "rgb(104,188,13)", "rgb(236,28,33)", 50);
 *
 * @instance
 *
 */
function promptContactlessLED(id, type, width, height, spacing, imageOn, imageOff, colorOn, colorOff, radius) {
    logger.debug('promptContactlessLED: id={}, type={}, width={}, height={}, spacing={}, imageOn={}, imageOff={}, colorOn={}, colorOff={}, radius={}', id, type, width, height, spacing, imageOn, imageOff, colorOn, colorOff, radius);
    contactlessLEDControl = {id: id, type: type, width: width, height: height, spacing: spacing, imageOn: imageOn, imageOff: imageOff, colorOn: colorOn, colorOff: colorOff, radius: radius};
    var state = nfc.getLedState();
    logger.debug('promptContactlessLED: Initial state={}', state);
    defaultNFCContactlessLEDCallback(state);
}

/**

 *  @description
 *	This Function update led panel with new state.
 *
 *  @param {number} state -  A 4 bit mask with a LED state
 *
 *  @example
 *	defaultNFCContactlessLEDCallback(15);
 *
 * @private
 *
 * @instance
 *
 */
function defaultNFCContactlessLEDCallback(state) {
    state = state.toString(2);
    while (state.length < 4) {
        state = '0' + state;
    }
    if (!contactlessLEDControl) {
        //if contactlessLEDControl is not present, discard event
        logger.debug('defaultNFCContactlessLEDCallback: LED state={}, no LED callback registered, exiting', state);
        return;
    }
    logger.debug('defaultNFCContactlessLEDCallback: LED state={}', state);
    var ledEl = document.getElementById(contactlessLEDControl.id);
    ledEl.innerHTML = null;
    for (var idx = 3; idx >= 0; idx--) {
        var ledChildEl = contactlessLEDControl.type === 'Generated' ? document.createElement('span') : document.createElement('img');
        ledChildEl.style.display = 'inline-block';
        ledChildEl.style.width = contactlessLEDControl.width + 'px';
        ledChildEl.style.height = contactlessLEDControl.height + 'px';
        ledChildEl.style.margin = contactlessLEDControl.spacing + 'px';
        if (contactlessLEDControl.type === 'Generated') {
            ledChildEl.style.borderRadius = contactlessLEDControl.radius + '%';
            if (state.charAt(idx) === '1') {
                ledChildEl.style.backgroundColor = contactlessLEDControl.colorOn;
            } else {
                ledChildEl.style.backgroundColor = contactlessLEDControl.colorOff;
            }
        } else {
            if (state.charAt(idx) === '1') {
                ledChildEl.src = contactlessLEDControl.imageOn;
            } else {
                ledChildEl.src = contactlessLEDControl.imageOff;
            }
        }
        ledEl.appendChild(ledChildEl);
    }
}

/**

 *  @description
 *	This function TODO missing description.
 *
 *  @param{string} domId
 *  @param {string} langCode - language code
 *  @param {boolean} langControlled - whether the image is language controlled
 *
 *  @example
 *  setupImage("ctrl_id", "$langCode")
 *
 * @instance
 *
 */
function setupImage(domId, langCode, langControlled) {
    logger.debug('setupImage: domId={} langCode={} langControlled={}', domId, langCode, langControlled);
    var imagesLength = document.getElementById(domId).querySelectorAll('img').length;
    if(imagesLength > 0) {
        displayImageAndStartTimer(domId, imagesLength, 0);
    }

    if (langControlled && langCode && langCode.charAt(0) === '$') {
        registerForVariable(langCode.substring(1), function () {
            displayLangControlledImage(domId, langCode);
        });
    }
    if (langControlled && langCode) {
        displayLangControlledImage(domId, langCode);
    }
}

/**

 *  @description
 *	This function display an image depending on system variable value.
 *
 *  @param {string} domId
 *  @param {string} langCode - language code
 *  @param {boolean} langControlled - whether the image is language controlled
 *  @param {string} variableName - system variable name
 *  @param {array} options - array of objects in format {name: ..., minValue: ..., maxValue: ...}
 *
 *  @example
 *  setupConditionalImage("ctrl_id", "$langCode", true, "$sysVar", [{name: viking.png, minValue: 0, maxValue: 100}])
 *
 * @instance
 *
 */
function setupConditionalImage(domId, langCode, langControlled, variableName, options) {
    logger.debug('setupConditionalImage: domId={} langCode={} langControlled={} variableName={} options={}', domId, langCode, langControlled, variableName, options);
    displayConditionalImage(domId, variableName, options);
    registerForVariable(variableName.substring(1), function () {
        displayConditionalImage(domId, variableName, options);
        if (langControlled && langCode) {
            displayLangControlledImage(domId, langCode);
        }
    });

    if (langControlled && langCode && langCode.charAt(0) === '$') {
        registerForVariable(langCode.substring(1), function () {
            displayLangControlledImage(domId, langCode);
        });
    }
    if (langControlled && langCode) {
        displayLangControlledImage(domId, langCode);
    }
}

/**
 * @description
 * TODO missing desctiption
 *
 * @param domId
 * @param variableName
 * @param options
 *
 * @private
 *
 * @instance
 *
 */

function displayConditionalImage(domId, variableName, options) {
    var variableValue = variableGet(variableName.substring(1));
    if (variableValue) {
        variableValue = Number(variableValue);
        for (var idx = 0; idx < options.length; idx++) {
            var option = options[idx];
            option.pathCorrection = option.pathCorrection || '';
            if ((!option.minValue && variableValue <= Number(option.maxValue)) || (!option.maxValue && variableValue >= Number(option.minValue)) || (variableValue >= Number(option.minValue) && variableValue <= Number(option.maxValue))) {
                document.getElementById(domId).innerHTML = null;
                var img = document.createElement('img');
                var imgPath = option.name.indexOf('/') == -1 ? option.pathCorrection + 'assets/images/' + option.name : option.name;
                img.src = imgPath;
                img.width = document.getElementById(domId).offsetWidth;
                document.getElementById(domId).appendChild(img);
                break;
            }
        }
    }
}

/**
 * @description
 * TODO missing description
 *
 * @param langCode
 *
 * @private
 *
 * @instance
 */

function setupPageBackgroundImage(langCode) {
    if (langCode && langCode.charAt(0) === '$') {
        registerForVariable(langCode.substring(1), function () {
            displayLangControlledImage(undefined, langCode);
        });
    }
    if (langCode) {
        displayLangControlledImage(undefined, langCode);
    }
}

/**
 * @description
 * TODO missing description
 *
 * @param divId
 * @param imagesLength
 * @param currentIndex
 *
 * @private
 *
 * @instance
 */

function displayImageAndStartTimer(divId, imagesLength, currentIndex) {
    if (currentIndex === imagesLength) {
        currentIndex = 0;
    }
    var divEl = document.getElementById(divId);
    var images = divEl.querySelectorAll('img');
    for (var i = 0; i < images.length; i++) {
        var img = images[i];
        img.style.display = 'none';
    }
    var image = divEl.querySelectorAll('img')[currentIndex];
    image.style.display = 'block';
    if (imagesLength && imagesLength > 1) {
        var duration = image.getAttribute('eq-duration') * 1000;
        setTimeout(function () {
            currentIndex += 1;
            displayImageAndStartTimer(divId, imagesLength, currentIndex);
        }, duration)
    }
}

/**
 * @description
 * TODO missing description
 *
 * @param domId
 * @param langCode
 *
 * @private
 *
 * @instance
 */

function displayLangControlledImage(domId, langCode) {
    if (!domId) {
        var imagePath = document.body.style.backgroundImage;
        var newPath = updateLangControlledPath(imagePath, langCode);
        if (imagePath !== newPath) {
            logger.debug('displayLangControlledImage: Page background image "{}" replaced with image "{}"', imagePath, newPath);
            document.body.style.backgroundImage = newPath;
        }
    } else {
        if(document.querySelectorAll('#' + domId + ' img').length) {
            var images = document.querySelectorAll('#' + domId + ' img');
            for (var i = 0; i < images.length; i++) {
                var newPath = updateLangControlledPath(images[i].src, langCode);
                if (images[i].src !== newPath) {
                    logger.debug('displayLangControlledImage: Image "{}" inside parent element with id={} replaced with image "{}"', images[i].src, domId, newPath);
                    images[i].src = newPath;
                }
            }
        } else {
            var bg = document.getElementById(domId).style.backgroundImage;
            if (bg) {
                var newPath = updateLangControlledPath(bg, langCode);
                if (bg !== newPath) {
                    logger.debug('displayLangControlledImage: Background image "{}" for element with id={} replaced with image "{}"', bg, domId, newPath);
                    document.getElementById(domId).style.backgroundImage = newPath;
                }
            }
        }
    }
}

/**
 * @description
 * TODO missing description
 *
 * @param filePath
 * @param langCode
 * @returns {*}
 *
 * @private
 *
 * @instance
 */

function updateLangControlledPath(filePath, langCode) {
    if (langCode && langCode.charAt(0) === '$') {
        langCode = variableGet(langCode.substring(1));
    }
    if (langCode) {
        var fileName = filePath.split('/').pop();
        fileName = fileName.substring(0, fileName.lastIndexOf('.'));
        var fileExtension = filePath.split('.').pop();
        var currentLangCode;
        if (fileName.indexOf('.') > -1) {
            currentLangCode = fileName.split('.').pop();
        }
        if (!currentLangCode || currentLangCode !== langCode) {
            fileName = currentLangCode ? fileName.substring(0, fileName.lastIndexOf('.')) : fileName;
            return filePath.substring(0, filePath.lastIndexOf('/') + 1) + fileName + '.' + langCode + '.' + fileExtension;
        }
    }
    return filePath;
}

/**
 * @description
 * TODO missing description
 *
 * @param signedImageName
 * @param langCode
 * @returns {*}
 *
 * @private
 *
 * @instance
 */

function updateLangControlledSignedImageName(signedImageName, langCode) {
    if (langCode && langCode.charAt(0) === '$') {
        langCode = variableGet(langCode.substring(1));
    }
    if (langCode) {
        var currentLangCode;
        if (signedImageName.indexOf('.') > -1) {
            signedImageName = signedImageName.substring(0, signedImageName.lastIndexOf('.'));
            currentLangCode = signedImageName.split('.').pop();
        }
        if (!currentLangCode || currentLangCode !== langCode) {
            return signedImageName + '.' + langCode;
        }
    }
    return signedImageName;
}

/**

 *  @description
 *	This function TODO missing description.
 *
 *  @param {string} domId
 *  @param {string} langCode - language code
 *  @param {boolean} langControlled - whether the video is language controlled
 *  @param {boolean} autoplay - whether the video should be autoplayed
 *
 *  @example
 *  setupVideo("ctrl_id", "$langCode", true, false)
 *
 * @instance
 *
 */
function setupVideo(domId, langCode, langControlled, autoplay) {
    logger.debug('setupVideo: domId={} langCode={} langControlled={} autoplay={}', domId, langCode, langControlled, autoplay);
    if (langControlled && langCode && langCode.charAt(0) === '$') {
        registerForVariable(langCode.substring(1), function () {
            var oldVideoEl = document.querySelector('#' + domId + ' video');
            var isOldVideoPlaying = (oldVideoEl.currentTime > 0) && !oldVideoEl.paused && !oldVideoEl.ended;
            displayLangControlledVideo(domId, langCode);
            if (isOldVideoPlaying){
                var videoSource = document.querySelector('#' + domId + ' video');
                videoSource.play();
            }
        });
    }
    if (langControlled && langCode) {
        displayLangControlledVideo(domId, langCode);
    }
    if (autoplay){
        document.querySelector('#' + domId + ' video').play();
    }
}

/**
 * @description
 * TODO missing description
 *
 * @param domId
 * @param langCode
 *
 * @private
 *
 * @instance
 */

function displayLangControlledVideo(domId, langCode) {
    var videoContainer = document.querySelector('#' + domId);
    if (videoContainer) {
        var srcUrl = videoContainer.querySelector('video source').src;
        var newPath = updateLangControlledPath(srcUrl, langCode);
        if (srcUrl !== newPath) {
            logger.debug('displayLangControlledVideo: Video source "{}" inside parent element with id={} replaced with source "{}"', srcUrl, domId, newPath);
            var oldVideoEl = videoContainer.querySelector('video');
            var newVideoEl = oldVideoEl.cloneNode([true])
            oldVideoEl.quit();
            newVideoEl.querySelector('source').src = newPath;
            // DOMElement.remove() is not working, using innerHTML to clear children
            videoContainer.innerHTML = '';
            videoContainer.appendChild(newVideoEl);
        }
    }
}



/*
 * ============================================================================================================
 *    								   	 jphoenix page functions
 * ============================================================================================================
 */

/**

 *  @description
 *  This function loads the page on terminal.
 *
 *  @param {string} path - is the relative (to home) location of html page
 *
 *  @example
 *  pageShow("viking/html/index.html");
 *
 * @instance
 *
 */
function pageShow(path) {
    logger.debug('pageShow: path={}', path);
    var statusCode = page.showPage(jPhoenixSettings.getBaseUrl() + 'page/show?path=' + path);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PAGE_SHOW_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *  This function loads the page on terminal.
 *
 *  @param {string} path - is the relative (to app's html directory) location of html page
 *
 *  @example
 *  pageShowRelative("index.html");
 *
 * @instance
 *
 */

function pageShowRelative(path) {
    logger.debug('pageShowRelative: path={}', path);
    var statusCode = page.showPage(jPhoenixSettings.getBaseUrl() + 'page/show?page=' + path);
    if (statusCode != 0) {
        throw new EQError(eqErrors.PAGE_SHOW_ERROR, {statusCode: statusCode});
    }
}

/*
 * ============================================================================================================
 *    								   	 jphoenix variable functions
 * ============================================================================================================
 */

/**

 *  @description
 *	This function will retrieve and return the named variable from the system using the URL http://localhost/cgi-bin/variable/get
 *	which in turn calls the CGI binary /home/variable/cgi-bin/get.
 *
 *  @param {string} varName - name of the variable
 *  @return {string} content - value for the name of the variable
 *
 * @instance
 *
 */
function variableGet(varName) {
    var content;
    content = registry.getValue(varName);
    if (content === null) {
        logger.info('variableGet: Unable to get variable {} value', varName);
    } else {
        logger.info("variableGet: varName={}, value={}", varName, content);
    }
    return content;
}

/**

 *  @description
 *	This function calls the internal cgi-bin URL http://localhost/cgi-bin/variable/set to set the value of one of more variables.
 *	Note:  To speed up the processing, the function supports a JSON formatted list of variables/values
 *	to allow for the setting of multiple variables with a single function call.  Format TBD.
 *
 *  @param {string} name - name of the variable
 *  @param {string | number | blob} value - value set to the name
 *
 * @instance
 *
 */
function variableSet(name, value) {
    var response = registry.setValue(name, value);
    //todo check response value for error
    logger.info("variableSet: varName={}, value={}, response={}", name, value, response);
}

/**

 *  @description
 *	This function copy value of variable varNameSrc to variable varNameDst.
 *
 *  @param {string} varNameSrc - name of the variable
 *  @param {string} varNameDst - name of the variable
 *
 * @instance
 *
 */
function variableCopy(varNameSrc, varNameDst) {
    logger.debug('variableCopy: varNameSrc={} varNameDst={}', varNameSrc, varNameDst);
    var varValue = variableGet(varNameSrc);
    variableSet(varNameDst, varValue);
}

var variableChangeCallbacks = {};
/**

 *  @description
 *	This function register callback functions that needs to be executed on variable value change.
 *  variableChangeCallbacks is map that contains  variableName -> [callbacks]
 *
 *  @param {string} variableName - name of the variable
 *  @param {function} callbackFunction - function with two parameters function(oldVal, newVal) { … } that needs to be executed on variable value change.
 *
 *  @example
 *  var regSysVar1Fn = function(oldVal, newVal) {...}
 *  registerForVariable('sysVar1', regSysVar1Fn);
 *
 * @instance
 *
 */
function registerForVariable(variableName, callbackFunction) {
    var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
    logger.debug('registerForVariable: variableName={}', variableName);
    if (!variableChangeCallbacks[lcVariableName]) {
        //first callback for given variable
        variableChangeCallbacks[lcVariableName] = [callbackFunction];
        //subscribe distinct variables just once
        registry.subscribe(variableName);
    } else {
        variableChangeCallbacks[lcVariableName].push(callbackFunction);
    }
}

/**

 *  @description
 *	This function removed specified callback from list of all callbacks for the given variable.
 *
 *  @param {string} variableName - name of the variable
 *  @param {function} callbackFunction - reference to function that needs to be removed from list of all registered callbacks for the given variable.
 *
 *  @example
 *  unregisterForVariable('sysVar1', regSysVar1Fn);
 *
 * @instance
 *
 */
function unregisterForVariable(variableName, callbackFunction) {
    var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
    logger.debug('unregisterForVariable: variableName={}', variableName);

    if (!callbackFunction) {
        logger.error('unregisterForVariable: Callback reference must be specified');
        return;
    }

    var varCallbacks = variableChangeCallbacks[lcVariableName];
    if (varCallbacks) {
        var callbackIdx = varCallbacks.indexOf(callbackFunction);
        if (callbackIdx > -1) {
            logger.debug('unregisterForVariable: Callback successfully unregistered for variable {}', variableName);
            if (varCallbacks.length == 1) {
                logger.debug('unregisterForVariable: Unregistering from registry for variable {}', variableName);
                registry.unsubscribe(variableName);
            }
            varCallbacks.splice(callbackIdx, 1);
            return;
        }
    }
    logger.error('unregisterForVariable: There is no callback registered for variable {}', variableName);
}

var variableDeleteCallbacks = {};
/**

 *  @description
 *	This function register callback functions that needs to be executed when variable is deleted.
 *  variableDeleteCallbacks is map that contains  variableName -> [callbacks]
 *
 *  @param {string} variableName - name of the variable
 *  @param {function} callbackFunction - function without parameters function() { … } that needs to be executed when variable is deleted.
 *
 *  @example
 *  var regSysVar1Fn = function() {...}
 *  registerForVariableDelete('sysVar1', regSysVar1Fn);
 *
 * @instance
 *
 */
function registerForVariableDelete(variableName, callbackFunction) {
    var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
    logger.debug('registerForVariableDelete: variableName={}', variableName);
    if (!variableDeleteCallbacks[lcVariableName]) {
        //first callback for given variable
        variableDeleteCallbacks[lcVariableName] = [callbackFunction];
        //subscribe distinct variables just once
        if (!variableChangeCallbacks[lcVariableName]) {
            registry.subscribe(variableName);
        }
    } else {
        variableDeleteCallbacks[lcVariableName].push(callbackFunction);
    }
}

/**

 *  @description
 *	This function removed specified callback from list of all callbacks for the given variable.
 *
 *  @param {string} variableName - name of the variable
 *  @param {function} callbackFunction - reference to function that needs to be removed from list of all registered callbacks for the given variable.
 *
 *  @example
 *  unregisterForVariableDelete('sysVar1', regSysVar1Fn);
 *
 * @instance
 *
 */
function unregisterForVariableDelete(variableName, callbackFunction) {
    var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
    logger.debug('unregisterForVariableDelete: variableName={}', variableName);

    if (!callbackFunction) {
        logger.error('unregisterForVariableDelete: Callback reference must be specified');
        return;
    }

    var varCallbacks = variableDeleteCallbacks[lcVariableName];
    if (varCallbacks) {
        var callbackIdx = varCallbacks.indexOf(callbackFunction);
        logger.debug("ima callbacl {}",variableDeleteCallbacks);
        if (callbackIdx > -1) {
            logger.debug('unregisterForVariableDelete: Callback successfully unregistered for variable {}', variableName);
            if (varCallbacks.length == 1 && !variableChangeCallbacks[lcVariableName]) {
                logger.debug('unregisterForVariableDelete: Unregistering from registry for variable {}', variableName);
                registry.unsubscribe(variableName);
            }
            varCallbacks.splice(callbackIdx, 1);
            return;
        }
    }
    logger.error('unregisterForVariableDelete: There is no callback registered for variable {}', variableName);
}

/*
 * ============================================================================================================
 *    								   	 jphoenix events and event listeners
 * ============================================================================================================
 */

var registeredEventsList = [];
var pageUpdateEvent;   //only one onUpdate is possible
/**

 *  @description
 *  This function registers listener for given event from given control.
 *  When event is fired, callbackFunction is called.
 *  Callback function signature should be function(name, text, response), where name is the name
 *  registeredEventsList @type {(Array)} - variable to save all the actions for events and their control names for future use
 *
 *  @param {string} eventName - name of the event to listen for: TODO: refactor to eventType
 *  @param {string} eventSourceName - name of event source
 *  @param {function} callbackFunction - callback that performs all the actions after an event received.
 *
 *  @example
 *  registerForEvent("onTouched", "confirm", function(name, text, response) {
 *      beep(1);
 *      processSignature();
 *  });
 *
 * @instance
 *
 */
function registerForEvent(eventName, eventSourceName, callbackFunction) {
    logger.debug('registerForEvent: eventName={}, eventSourceName={}', eventName, eventSourceName);

    if(eventName === "onUpdate") {
        pageUpdateEvent = {eventSourceName: eventSourceName, callbackFunction: callbackFunction};
    } else {
        for (var i = 0; i < registeredEventsList.length; i++) {
            if (eventSourceName === registeredEventsList[i].eventSourceName && eventName === registeredEventsList[i].eventName) {
                registeredEventsList[i].actions = callbackFunction;
                return;
            }
        }
        registeredEventsList.push({eventSourceName: eventSourceName, eventName: eventName, actions: callbackFunction});
    }
}

/**

 *  @description
 *  This function unregisters listener for given event from given control.
 *  registeredEventsList @type {(Array)} - variable to save all the actions for events and their control names for future use
 *
 *  @param {string} eventName - name of the event to listen for: TODO: refactor to eventType
 *  @param {string} eventSourceName - name of event source
 *
 *  @example
 *  unregisterForEvent("onTouched", "confirm");
 *
 * @instance
 *
 */
function unregisterForEvent(eventName, eventSourceName) {
    logger.debug('unregisterForEvent: eventName={}, eventSourceName={}', eventName, eventSourceName);

    if(eventName === "onUpdate") {
        pageUpdateEvent = null;
    } else {
        for (var i = 0; i < registeredEventsList.length; i++) {
            if (eventSourceName === registeredEventsList[i].eventSourceName && eventName === registeredEventsList[i].eventName) {
                registeredEventsList.splice(i);
                return;
            }
        }
    }
}

/**

 *  @description
 *  This activates all touch input areas defined by promptTouchArea(), promptCheckbox(), promptRadiobox() etc...
 *  Event received may contain following:
 *      events: touchArea, checkBox, radioBox
 *      data: signature, textBox, maualPAN, manualCVV2, expiryDate, checkBox, radioBox
 *
 *  @param {string} timeout - time to wait for the events from the terminal
 *
 *  @example
 *  promptEvent( 20 );
 *
 * @instance
 *
 */
function promptEvent(timeout) {
    logger.debug('promptEvent: timeout={}', timeout);
    if (typeof timeout == "undefined") {
        timeout = -1;
    }
    //promptEvent( int timeout, bool isAutomaticRepeatRequired = false, bool keepSecureScreen = false )
    var responseData = secure.promptEvent(timeout);
    //var responseData = secure.promptEvent(timeout, true, false);  //was in 1.9.6 and earlier
    if (responseData && responseData.status == 0) {
        logger.debug('promptEvent: responseData={}', responseData);
    } else {
        throw new EQError(eqErrors.PROMPT_EVENT_ERROR, responseData);
    }
}

/**

 *  @description
 *  This activates all touch input areas defined by promptTouchArea(), promptCheckbox(), promptRadiobox() etc...
 *  Event received may contain following:
 *      events: touchArea, checkBox, radioBox
 *
 *  @param {string} timeout - time to wait for the events from the terminal
 *
 *  @example
 *  promptEventOnly( 20 );
 *
 * @instance
 *
 */
function promptEventOnly(timeout) {
    logger.debug('promptEventOnly: timeout={}', timeout);
    if (typeof timeout == "undefined") {
        timeout = -1;
    }
    var responseData = secure.promptEventOnly(timeout);
    if (responseData && responseData.status == 0) {
        logger.debug('promptEventOnly: responseData={}', responseData);
    } else {
        throw new EQError(eqErrors.PROMPT_EVENT_ERROR, responseData);
    }
}

/**

 *  @description
 * //TODO missing description.
 *
 *  @param {string} eventName - event name
 *  @param {string} eventId - control name or event id
 *  @param {(string | xmlObject)} data - response data
 *
 *  @private
 *
 * @instance
 *
 */
function executeCallBackFunction(eventName, eventId, data) {
    for (var i = 0; i < registeredEventsList.length; i++) {
        if (registeredEventsList[i].eventSourceName === eventId && registeredEventsList[i].eventName === eventName) {
            registeredEventsList[i].actions(eventId, data, data);
            break;
        }
    }
}

/**

 *  @description
 *	This callback is called from promptEvent() function onSuccess.
 *
 *  @param {(string | xmlObject)} data - success response from promptEvent()
 *
 *  @private
 *
 * @instance
 *
 */
function callBackFunction(data) {
    if (typeof data === 'object') {
        if(data['__SERVICEID__'] === 'com.equinoxpayments.p2pe') {  //this is marker for msr event data
            //handling msr event

            /*
             data in clear, keys: __SERVICEID__ , cardRangeId, keyId, keyType, ktb, panEncrypted, panHash, panMasked, track1, track2, track3, typeId
             data encrypted, keys: __SERVICEID__ , cardRangeId, keyId, keyType, ktb, panEncrypted, panHash, panMasked, track1Encrypted, track1Masked, track2Encrypted, track2Masked, track3Encrypted, track3Masked, typeId
             data error, keys: error?
             */

            var resultData;

            //NOTE: msr errors are handled in p2pe.dataError.connect(...) handler
            if(!data["error"]) {
                var missingTracks = [];
                msrMinTracksReadForValidity.forEach(function (trackId) {
                    if (!(data['track' + trackId] || data['track' + trackId + "Encrypted"] || data['track' + trackId + "Masked"])) {
                        missingTracks.push(trackId);
                    }
                });
                if (missingTracks.length) {
                    throw new EQError(eqErrors.MSR_TRACK_NOT_FOUND, {missingTracks: missingTracks});
                } else {
                    logger.debug('callBackFunction: All tracks from msrMinTracksReadForValidity are received');
                    resultData = data;

                    delete resultData.__SERVICEID__;        //don't need this in data. delete it

                    var tracksToDelete = [1, 2, 3].filter(function (x) {
                        return msrTracksToRead.indexOf(x) < 0
                    });
                    logger.debug("callBackFunction: msrTracksToRead={}, tracksToDelete={}", msrTracksToRead, tracksToDelete);
                    //delete unwanted tracks
                    tracksToDelete.forEach(function (trackId) {
                        logger.warn('callBackFunction: deleting track{} data (potentially non-existent)', trackId);
                        delete resultData["track" + trackId];
                        delete resultData["track" + trackId + "Encrypted"];
                        delete resultData["track" + trackId + "Masked"];
                    });
                }
            }
            for (var i = 0; i < registeredEventsList.length; i++) {
                if (registeredEventsList[i].eventSourceName === "msr" && registeredEventsList[i].eventName === "onMsr") {
                    registeredEventsList[i].actions("msr", resultData, resultData);
                    break;
                }
            }
        } else {
            Object.keys(data).forEach(function (key) {
                var name = key;
                var text = data[key];
                for (var i = 0; i < registeredEventsList.length; i++) {
                    if (registeredEventsList[i].eventSourceName === name) {
                        registeredEventsList[i].actions(name, text, data);
                    }
                }
            });
        }
    } else {
        if (data && data.indexOf('TIMEOUT') != -1) {
            //do something when timeout occurs
            return;
        }
        var specialHandling = null;
        if (typeof data === 'string' && data != 'canceled') {
            var prompts = stringToXMLDom(data).getElementsByTagName("prompt");
            for (var idx = 0; idx < prompts.length; idx++) {
                try {
                    var prompt = prompts[idx];
                    var name = prompt.getElementsByTagName('name')[0].childNodes[0].nodeValue;
                    var text = prompt.getElementsByTagName('text')[0].childNodes[0].nodeValue;
                    // trigger appropriate callback
                    for (var i = 0; i < registeredEventsList.length; i++) {
                        if (registeredEventsList[i].eventSourceName === name) {
                            registeredEventsList[i].actions(name, text, data);
                            specialHandling = 1;
                        }
                    }
                } catch (e) {
                    logs.log('Error parsing secure input: ' + e.message + ', data: ' + data);
                }
            }
            //promptEvent();

            if (specialHandling == null) {
                // Treat this as an XML and call key handlers if any
                var xml = stringToXMLDom(data);
                if (xml) {
                    var x = xml.getElementsByTagName("*");
                    for (var i = 0; i < x.length; ++i) {
                        var key = x[i].nodeName;
                        var val = null;
                        if (x[i].firstChild != null) {
                            val = x[i].firstChild.nodeValue;
                        }
                        for (var j = 0; j < registeredEventsList.length; ++j) {
                            if (registeredEventsList[j].eventSourceName === key) {
                                registeredEventsList[j].actions(key, val, xml);
                            }
                        }
                    }
                }
            }
        }
    }
}

/**

 *  @description
 *	This function adds an event to an event queue that a waiting application can read by calling the URL http://URL/cgi-bin/page/event.
 * 	Note that the calling application my be resident on the Luxe terminal or external to the terminal.
 *	The event name, type and ID are all determined from the element that called the function.
 *
 *  @param {*} response - could be any event, text
 *
 * @instance
 *
 */
function fireEvent(response) {
    logger.debug('fireEvent: response={}', response);
    page.fireEvent(response);  //fireEvent is implemented in page (WebKit Bridge) object
}

/**

 *  @description
 *  todo This function needs to be refactored or eliminated in future.
 *
 *  @private
 *  @deprecated
 *
 * @instance
 *
 */
function fireXmlEvent(x) {
    $.get('/cgi-bin/page/.fireEvent', {RAW_XML: x}, "text").done(function (data) {
        console.log('from fireXMLEvent', data);
    });
}

/**

 *  @description
 *  TODO missing description.
 *
 *  @property {string} parseXML - parses the xml data and appends text to DOM elements
 *
 * @private
 *
 */
var divBoxXMLParser = {
    parseXML: function (data) {
        var xmlDoc = stringToXMLDom(data);
        var updates = xmlDoc.getElementsByTagName("update");
        for (var idx = 0; idx < updates.length; idx++) {
            var update = updates[idx];
            var updateAction = update.getAttribute('action');
            var updateId = update.getAttribute('id');
            var children = update.childNodes;
            for (j = 0; j < children.length; j++) {
                if (children[j].nodeName == "text") {
                    updateText = children[j].textContent;
                    var parentEl = document.getElementById(updateId);
                    if (updateAction == "append") {
                        parentEl.innerHTML += updateText;
                        updateScroll(updateId);
                    } else if (updateAction == "prepend") {
                        parentEl.innerHTML = updateText + parentEl.innerHTML;
                    } else if (updateAction == "replace") {
                        parentEl.innerHTML = updateText;
                    } else if (updateAction == "delete") {
                        parentEl.innerHTML = null;
                    }
                }
            }
        }
    }
};

/*
 * ============================================================================================================
 *    								        on page load
 * ============================================================================================================
 */

/**
 *  @description
 *	Connect Secure Input signals just once.
 *
 * @private
 *
 * @instance
 *
 */
(function () {
    logger.debug("on page load, registering event listeners: page={}", window.location.pathname);

    page.onPageUpdate.connect(function (data) {
        if (pageUpdateEvent) {
            logger.debug("page.onPageUpdate event arrived: data='{}'", data);
            pageUpdateEvent.callbackFunction(pageUpdateEvent.eventSourceName, data, data);
        } else {
            logger.warn("page.onPageUpdate event arrived, onUpdate event not set !: data='{}'", data);
        }
    });
    secure.onSecureInput.connect(function (data) {
        logger.debug("secure.onSecureInput event arrived: data='{}'", data);
        if (data) {
            callBackFunction(data);
        }
    });
    /*
    secure.onSecureInput2.connect(function (data) {
        logger.debug("secure.onSecureInput2 event arrived: data='{}'", data);
        try {
            logger.debug("secure.onSecureInput2 event arrived: data='{}'", data);
            if (data) {
                var properties = Object.keys(data);
                for (var idx = 0; idx < properties.length; idx++) {
                    var property = properties[idx];
                    if (property !== 'eventSource') {
                        var ctrlData = data[property];
                        if (ctrlData && ctrlData.type === 'button') {
                            ctrlData.eventSource = data.eventSource;
                            executeCallBackFunction("onTouchOrKey", "touchOrKey", ctrlData);
                        }
                    }
                }
            }
        } catch (e) {
            logs.log(e.message);
        } finally {
        }
    });
    */

    // Receive PIN entry results
    secure.onPinEntryCompleted.connect(function (data) {
        try {
            logger.debug("secure.onPinEntryCompleted  event arrived: data='{}'", data);
            if (data) {
                callBackFunction({"pinblock": data});
            }
        } catch (e) {
            logs.log(e.message);
        } finally {
        }
    });

    //	Variable value change
    registry.valueChanged.connect(function (variableName, newVal) {
        logger.debug("registry.valueChanged  event arrived: variableName={}, newVal='{}'", variableName, newVal);
        //NOTE: temporary workaround for https://jira.eqpmt.net/browse/PHX-1068
        if ((newVal != null) && (newVal.length == 1) && (newVal.charCodeAt(0) == 0)) {
            newVal = "";
        }
        var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
        var vcList = variableChangeCallbacks[lcVariableName];
        if (vcList) {
            vcList.forEach(function (callback) {
                callback(null, newVal); //currently platform doesn't provide oldValue
            });
        }
    });

    //	Variable deleted
    registry.keyDeleted.connect( function(variableName) {
        logger.debug("registry.keyDeleted  event arrived: variableName={}", variableName);
        var lcVariableName = variableName.toLowerCase();  //registry variables are case insensitive
        var vcList = variableChangeCallbacks[lcVariableName];
        if (vcList) {
            vcList.forEach(function (callback) {
                callback(null, null); //Send null as new value. Currently platform doesn't provide oldValue
            });
        }
        var vdList = variableDeleteCallbacks[lcVariableName];
        if (vdList) {
            vdList.forEach(function (callback) {
                callback();
            });
        }
    });

    // NFC change state
    nfc.ledState.connect(function (state) {
        logger.debug("nfc.ledState  event arrived: state='{}'", state);
        defaultNFCContactlessLEDCallback(state);
    });

    // p2pe manualPAN and manualCVV2
    p2pe.plainPAN.connect(function (sourceId, data) {
        logger.debug("p2pe.plainPAN event arrived: sourceId={} data={}", sourceId, JSON.stringify(data));
        delete data.__SERVICEID__;        //don't need this in data. delete it
        executeCallBackFunction("onReturn", "manualPAN", data);
    });
    p2pe.encryptedPAN.connect(function (sourceId, data) {
        logger.debug("p2pe.encryptedPAN event arrived: sourceId={} data={}", sourceId, JSON.stringify(data));
        delete data.__SERVICEID__;        //don't need this in data. delete it
        executeCallBackFunction("onReturn", "manualPAN", data);
    });
    p2pe.plainCVV2.connect(function (sourceId, data) {
        logger.debug("p2pe.plainCVV2 event arrived: sourceId={} data={}", sourceId, JSON.stringify(data));
        delete data.__SERVICEID__;        //don't need this in data. delete it
        executeCallBackFunction("onReturn", "manualCVV2", data);
    });
    p2pe.encryptedCVV2.connect(function (sourceId, data) {
        logger.debug("p2pe.encryptedCVV2 event arrived: sourceId={} data={}", sourceId, JSON.stringify(data));
        delete data.__SERVICEID__;        //don't need this in data. delete it
        executeCallBackFunction("onReturn", "manualCVV2", data);
    });
    p2pe.dataError.connect(function (sourceId, errorId, data) {
        logger.debug("p2pe.dataError event arrived: sourceId={} errorId={} data={}", sourceId, errorId, JSON.stringify(data));
        if (sourceId == 1) {
            //handle only msr errors
            resultData = {
                errorCode: eqErrors.MSR_GENERAL_ERROR.errorCode,
                errorMsg: eqErrors.MSR_GENERAL_ERROR.errorMessage,
                subComponentErrorDetails: {sourceId: sourceId, errorId: errorId, data: data}
            };

            executeCallBackFunction("onMsr", "msr", resultData);
        } else if (sourceId == 2) {
            //handle only manualPAN/CVV errors
            resultData = {
                errorCode: eqErrors.MANUALPANCVV2_GENERAL_ERROR.errorCode,
                errorMsg: eqErrors.MANUALPANCVV2_GENERAL_ERROR.errorMessage,
                subComponentErrorDetails: {sourceId: sourceId, errorId: errorId, data: data}
            };

            executeCallBackFunction("onReturn", _activeManualPrompt, resultData);
        } else {
            logger.warn("p2pe.dataError: Unsupported sourceId {}", sourceId);
        }
    });
    scr.inserted.connect(function () {
        var status = scr.getATR();
        logger.debug("scr.inserted event arrived: status={}", JSON.stringify(status));
        var resultData = status;
        if (!status || status.status) {
            // TODO Check if throw error is needed
            var error = eqErrors.CARD_INSERTED_ERROR;
            logger.error("scr.inserted: errorCode={} errorMessage={}", error.errorCode, error.errorMessage);
            resultData = {
                errorCode: error.errorCode,
                errorMsg: error.errorMessage,
                subComponentErrorDetails: {sourceId: undefined, errorId: undefined, data: status}
            };
        } else if (status.atr && status.atr.length) {
            var atrHexString = '';
            for (var idx = 0; idx < status.atr.length; idx++) {
                var hex = status.atr[idx].toString(16);
                atrHexString += ("0" + hex).slice(-2);
            }
            resultData.atrHexString = atrHexString;
        }
        executeCallBackFunction("onCardInserted", "cardInserted", resultData);
    });
    scr.removed.connect(function (data) {
        logger.debug("scr.removed event arrived: data={}", JSON.stringify(data));
        executeCallBackFunction("onCardRemoval", "cardRemoval", data);
    });
    nfc.onPreDeactivate.connect(function (withRemoval) {
        logger.debug("nfc.onPreDeactivate arrived: withRemoval={}", withRemoval);
        if (withRemoval) {
            executeCallBackFunction("onNFCCardRemoval", "nfcCardRemoval", withRemoval);
        }
    });
    nfc.onUIRequest.connect(function (data) {
        logger.debug("nfc.onUIRequest event arrived: data={}", data ? JSON.stringify(data) : data);
        if (data) {
            var treeId = 1;
            utils.configtree_Clear(treeId);
            var status = utils.configtree_ToTree(treeId, data);
            if (status == 0) {
                var i;
                var langPrefData = utils.configtree_GetBinValue(treeId, "UIREQDATA.LANGPREF").value;
                var langPref = null;
                if (langPrefData) {
                    langPref = [];
                    for (i = 0; i < langPrefData.length; i++) {
                        langPref.push(langPrefData[i]);
                    }
                }

                var valueData = utils.configtree_GetBinValue(treeId, "UIREQDATA.VALUE").value;
                var value = null;
                if (valueData) {
                    value = [];
                    for (i = 0; i < valueData.length; i++) {
                        value.push(valueData[i]);
                    }
                }

                var currCodeData = utils.configtree_GetBinValue(treeId, "UIREQDATA.CURRCODE").value;
                var currCode = null;
                if (currCodeData) {
                    currCode = [];
                    for (i = 0; i < currCodeData.length; i++) {
                        currCode.push(currCodeData[i]);
                    }
                }

                /*
                 msgId as int
                 status as int
                 holdTime as int
                 langPref as array (8 byte array)
                 qval as int
                 valueas as array (6 byte array)
                 currCode as array (2 byte array)
                 */
                var jsonData = {
                    msgId: utils.configtree_GetIntValue(treeId, "UIREQDATA.MSGID").value,
                    status: utils.configtree_GetIntValue(treeId, "UIREQDATA.STATUS").value,
                    holdTime: utils.configtree_GetIntValue(treeId, "UIREQDATA.HOLDTIME").value,
                    langPref: langPref,
                    qval: utils.configtree_GetIntValue(treeId, "UIREQDATA.QVAL").value,
                    value: value,
                    currCode: currCode
                };
                executeCallBackFunction('onNFCUIRequest', 'nfcUIRequest', jsonData);
            } else {
                throw new EQError(eqErrors.NFC_ON_UI_REQUEST_DATA_ERROR, {status: status});
            }
        }
    });
    nfc.onPluginProcessed.connect(function (data) {
        logger.debug("nfc.onPluginProcessed event arrived: data={}", data ? JSON.stringify(data) : data);
        if (data) {
            var treeId = 2;
            utils.configtree_Clear(treeId);
            var status = utils.configtree_ToTree(treeId, data);
            var jsonData;
            if (status == 0) {
                var androidErrorCode = utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.error.code).value;
                if(androidErrorCode){
                    jsonData = {
                        code: androidErrorCode,
                        applicable: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.error.applicable).value,
                        status: utils.configtree_GetStringValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.error.status).value,
                        state: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.error.state).value,
                        sw: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.error.sw).value
                    };
                    throw new EQError(eqErrors.NFC_ON_PLUGIN_PROCESSED_ANDROID, {androidPay: jsonData});
                }
                var appleErrorCode = utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.error.code).value;
                if(appleErrorCode){
                    jsonData = {
                        code: appleErrorCode,
                        applicable: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.error.applicable).value,
                        status: utils.configtree_GetStringValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.error.status).value,
                        state: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.error.state).value,
                        sw: utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.error.sw).value
                    };
                    throw new EQError(eqErrors.NFC_ON_PLUGIN_PROCESSED_APPLE, {applePay: jsonData});
                }
                var version = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.androidPay.version);
                if(version) {
                    jsonData = nfc.utils.onPluginProcessedAndroidPay(treeId);
                    jsonData.androidPay.version = version;
                } else {
                    jsonData = nfc.utils.onPluginProcessedApplePay(treeId);
                }
                executeCallBackFunction("onNFCPluginProcessed", "nfcPluginProcessed", jsonData);
            } else {
                throw new EQError(eqErrors.NFC_ON_PLUGIN_PROCESSED_GENERAL, {status: status});
            }
        }
    });
    nfc.utils = {
        onPluginProcessedAndroidPay: function(treeId) {
            jsonData = {
                androidPay:{}
            };
            jsonData.androidPay.merchant = {};
            jsonData.androidPay.status = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.androidPay.status);
            jsonData.androidPay.merchant.id = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.androidPay.merchId);
            jsonData.androidPay.merchant.service = {};
            var i = 1;
            while (true) {
                var serviceValue = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.androidPay.merchService + "." + i);
                if(serviceValue === null){
                    break;
                }
                var j = 0;
                jsonData.androidPay.merchant.service[i] = {};
                jsonData.androidPay.merchant.service[i].value = serviceValue;
                var serviceRecStr;
                while(true) {
                    serviceRecStr = "." + i + "." + j;
                    var recId = utils.configtree_GetStringValue(treeId, nfc.utils.onPlgProcessedKeys.androidPay.merchService + serviceRecStr + ".ID").value;
                    if(recId === "") {
                        break;
                    }
                    jsonData.androidPay.merchant.service[i][j] = {};
                    jsonData.androidPay.merchant.service[i][j].ID = recId;
                    var keyId = null;
                    if (recId.indexOf('ly') > -1) {
                        keyId = 'ly';
                    } else if (recId.indexOf('of') > -1) {
                        keyId = 'of';
                    } else if (recId.indexOf('gc') > -1) {
                        keyId = 'gc';
                    } else if (recId.indexOf('pl') > -1) {
                        keyId = 'pl';
                    } else if (recId.indexOf('cus') > -1) {
                        keyId = 'cus';
                    }
                    if(keyId){
                        for(var x = 0; x < nfc.utils.onPlgProcessedKeys.androidPay[keyId].length; x++){
                            var key = nfc.utils.onPlgProcessedKeys.androidPay[keyId][x];
                            var keyData = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.androidPay.merchService + serviceRecStr + "." + key);
                            if(keyData){
                                jsonData.androidPay.merchant.service[i][j][key] = keyData;
                            }
                        }
                    }
                    j++;
                }
                i++;
            }
            return jsonData;
        },
        onPluginProcessedApplePay: function(treeId) {
            var i = 0;
            jsonData = {
                applePay:{}
            };
            jsonData.applePay.merch = {};
            while(true){
                var merchId = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.applePay.merch + "." + i + ".ID");
                if(merchId === null) {
                    break;
                }
                jsonData.applePay.merch[i] = {};
                jsonData.applePay.merch[i].id = merchId;
                jsonData.applePay.merch[i].sw = utils.configtree_GetIntValue(treeId, nfc.utils.onPlgProcessedKeys.applePay.merch + "." + i + ".SW").value;
                jsonData.applePay.merch[i].token = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.applePay.merch + "." + i + ".TOKEN");
                var data = nfc.utils.extractBinData(treeId, nfc.utils.onPlgProcessedKeys.applePay.merch + "." + i + ".DATA");
                if(data){
                    jsonData.applePay.merch[i].data = data;
                }
                // todo data_decrypted
                i++;
            }
            return jsonData;
        },
        extractBinData: function(treeId, key) {
            var binData = utils.configtree_GetBinValue(treeId, key).value;
            var arrayData = null;
            if(binData){
                arrayData = [];
                for (i = 0; i < binData.length; i++) {
                    arrayData.push(binData[i]);
                }
            }
            return arrayData;
        },
        onPlgProcessedKeys: {
            androidPay: {
                status: "STATUS",
                version: "OUTCOME.PLUGINS.ANDROIDPAY.VERSION",
                merchId: "OUTCOME.PLUGINS.ANDROIDPAY.MERCHANT.ID",
                merchService: "OUTCOME.PLUGINS.ANDROIDPAY.MERCHANT.SERVICE",
                ly: [ // must contain oid, n, may contain tr1, tr2
                    "oid","n","tr1","tr2"
                ],
                of: [ // must contain oid, n
                    "oid","n"
                ],
                gc: [ // must contain oid, n, may contain tr1, tr2, p
                    "oid","n","tr1","tr2","p"
                ],
                pl: [ // must contain oid, n, may contain tr1, tr2, p, c1, ex
                    "oid","n","tr1","tr2","p","cl","ex"
                ],
                cus: [ // must contain cid, may contain cpl, cut, cud
                    "cid","cpl","cut","cud"
                ],
                error: {
                    applicable: "OUTCOME.PLUGINS.ANDROIDPAY.ERROR.APPLICABLE",
                    code: "OUTCOME.PLUGINS.ANDROIDPAY.ERROR.CODE",
                    state: "OUTCOME.PLUGINS.ANDROIDPAY.ERROR.STATE",
                    status: "OUTCOME.PLUGINS.ANDROIDPAY.ERROR.STATUS",
                    sw: "OUTCOME.PLUGINS.ANDROIDPAY.ERROR.SW"
                }
            },
            applePay: {
                merch: "OUTCOME.PLUGINS.APPLEPAY.MERCH",
                error:{
                    applicable:"OUTCOME.PLUGINS.APPLEPAY.ERROR.APPLICABLE",
                    code:"OUTCOME.PLUGINS.APPLEPAY.ERROR.CODE",
                    state:"OUTCOME.PLUGINS.APPLEPAY.ERROR.STATE",
                    status:"OUTCOME.PLUGINS.APPLEPAY.ERROR.STATUS",
                    sw:"OUTCOME.PLUGINS.APPLEPAY.ERROR.SW"
                }
            }
        }
    }
    window.onkeydown = function (e) {
        e = window.event || e;
        logger.debug("window.onkeydown event arrived: keyCode={}", e.keyCode);
        executeCallBackFunction("onKeypad", "keypad", e.keyCode);
    };
})();

/*
 * ============================================================================================================
 *    										jphoenix actions
 * ============================================================================================================
 */

/**

 *  @description
 *	This function makes beep sounds with specific frequency, length of a beep, time gap between each beep.
 *
 *  @param {number} repeat - number of times to beep sound
 *  @param {number} freq - frequency number
 *  @param {number} length - length of a beep in milliseconds
 *  @param {number} delay - delay time gap between each beep sound in milliseconds
 *  @return {string} success data
 *
 * @instance
 *
 */
function beepEx(repeat, freq, length, delay) {
    logger.debug('beepEx: repeat={}, freq={}, length={}, delay={}', repeat, freq, length, delay);
    var statusCode = tools.beep(repeat, freq, length, delay);
    if (statusCode != 0) {
        throw new EQError(eqErrors.ACTION_BEEP_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function makes beep sounds.
 *
 *  @param {number} numberOfBeeps - number of times to beep
 *  @return {string} success data
 *
 * @instance
 *
 */
function beep(numberOfBeeps) {
    logger.debug('beep: numberOfBeeps={}', numberOfBeeps);
    var statusCode = tools.beep(numberOfBeeps, 450, 50, 50);
    if (statusCode != 0) {
        throw new EQError(eqErrors.ACTION_BEEP_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function set speaker volume.
 *
 *  @param {number} volume
 *  @return {string} success data
 *
 * @instance
 *
 */
function setSpeakerVolume(volume) {
    logger.debug('setSpeakerVolume: volume={}', volume);
    var statusCode = tools.setSpeakerVolume(volume);
    if (statusCode == -1) {
        throw new EQError(eqErrors.ACTION_SET_SPEAKER_VOLUME_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function set speaker volume.
 *
 *  @return {string} speaker volume
 *
 * @instance
 *
 */
function getSpeakerVolume() {
    var response = tools.getSpeakerVolume();
    if (response == -1) {
        throw new EQError(eqErrors.ACTION_GET_SPEAKER_VOLUME_ERROR, {response: response});
    } else {
        logger.debug('getSpeakerVolume: volume={}', response);
    }
}

/**

 *  @description
 *	This function runs binary async (doesnt wait for binary execution to complete).
 *
 *  @param {string} application - application whose binary to execute
 *  @param {string} binary - binary whose binary to execute from /home/<application/bin/* directory
 *  @param {string} parameters - command line parameters to pass to binary
 *
 *  @example
 *  runProcess("moneris", "runEMVTransaction", "-i abc -j def");
 *
 * @instance
 *
 */
function runProcess(application, binary, parameters){
    logger.debug('runProcess: application={}, binary={}, parameters={}', application, binary, parameters);
    var statusCode = tools.runProcessAsync(application, binary, parameters);
    if (statusCode == -1) {
        throw new EQError(eqErrors.ACTION_RUN_PROCESS_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function log messages into log file.
 *
 *  @param {string} msg - to send to log file
 *
 *  @example
 *  logDebug("this is a test msg");
 *
 * @instance
 *
 */
function logDebug(msg) {
    logger.debug(msg);
}

/**

 *  @description
 *	This function stop timer.
 *
 *  @param {string} name - name of the timer to stop
 *
 *  @example
 *  stopTimer("testTimer");
 *
 * @instance
 *
 */
function stopTimer(name) {
    var timer = registeredTimers[name];
    if (!timer) {
        logger.error("Timer {} does not exist", name);
        return;
    }
    logger.debug('stopTimer: name={} type={} intervalInMS={} timerId={}', name, timer.type, timer.intervalInMS, timer.timerId);
    if (timer.type === 'interval') {
        clearInterval(timer.timerId);
    } else if (timer.type === 'onetime') {
        clearTimeout(timer.timerId);
    }
}

/**

 *  @description
 *	This function start timer.
 *
 *  @param {string} name - name of the timer to start
 *
 *  @example
 *  startTimer("testTimer");
 *
 * @instance
 *
 */
function startTimer(name) {
    var timer = registeredTimers[name];
    if (!timer) {
        logger.error("Timer {} does not exist", name);
        return;
    }
    logger.debug('startTimer: name={} type={} intervalInMS={}', name, timer.type, timer.intervalInMS);
    if (timer.type === 'interval') {
        timer.timerId = setInterval(timer.callbackFn, timer.intervalInMS)
    } else if (timer && timer.type === 'onetime') {
        timer.timerId = setTimeout(timer.callbackFn, timer.intervalInMS);
    }
}

var registeredTimers = {};
/**

 *  @description
 *  This function registers timers.
 *  When intervalInMS is reached, callbackFn is called.
 *
 *  @param {string} name - name of the timer
 *  @param {string} type - timer type (interval or onetime)
 *  @param {number} intervalInMS - execute all actions once or periodically after intervalInMS
 *  @param {function} callbackFn - callback that performs all the actions after intervalInMS.
 *
 *  @example
 *  registerTimer("Timer Test", "interval", 100, function() {
 *      beep(1);
 *      processSignature();
 *  });
 *
 *  @private
 *
 * @instance
 *
 */
function registerTimer(name, type, intervalInMS, callbackFn) {
    logger.debug('registerTimer: name={}, type={} intervalInMS={}', name, type, intervalInMS);
    registeredTimers[name] = {name: name, type: type, intervalInMS: intervalInMS, callbackFn: callbackFn};
}

/**

 *  @description
 *  This function toggle control visibility depending on variable value.
 *
 *  @param {string} id - control id
 *  @param {string} id - control id
 *  @param {string} hiddenVariableName - variable that determines whether the control should be visible
 *  @param {number} hiddenVariableValue - variable value
 *  @param {boolean} isInputControl - enable or disable touch area for all input controls
 *  @param {boolean} hideIfVariableIsEmpty - hide control if variable value is empty or not exist
 *  @param {string} hiddenOrVisible - define is control visible or hidden if variable value condition is satisfied. Possible values are 'hidden' and 'visible'
 *
 *  @private
 *
 * @instance
 *
 */
function toggleControlVisibility(id, name, hiddenVariableName, hiddenVariableValue, isInputControl, hideIfVariableIsEmpty, hiddenOrVisible) {
    logger.debug('toggleControlVisibility: id={}, name={}, hiddenVariableName={}, hiddenVariableValue={}, isInputControl={}, hideIfVariableIsEmpty={}, hiddenOrVisible={}', id, name, hiddenVariableName, hiddenVariableValue, isInputControl, hideIfVariableIsEmpty, hiddenOrVisible);
    if (hiddenVariableValue.charAt(0) === '$') {
        hiddenVariableValue = hiddenVariableValue.substring(1);
        hiddenVariableValue = variableGet(hiddenVariableValue);
    }
    var variableValue;
    if (hiddenVariableName) {
        variableValue = variableGet(hiddenVariableName);
    }

    var controlElement = document.getElementById(id);
    var hidden = false;
    if (hiddenOrVisible === 'hidden' && (hideIfVariableIsEmpty && !variableValue || variableValue === hiddenVariableValue)) {
        hidden = true;
    } else if(hiddenOrVisible === 'visible' && (hideIfVariableIsEmpty && !variableValue || variableValue === hiddenVariableValue)) {
        hidden = false;
    } else if (hiddenOrVisible === 'visible') {
        hidden = true;
    }
    if (controlElement) {
        //hide/show html rendered control portion
        var ctrlClass = controlElement.getAttribute('eq-ctrl-class');
        if (ctrlClass === 'radiobox'){
            controlElement.style.opacity = hidden ? 0.5 : 1;
        } else {
            controlElement.style.visibility = hidden ? 'hidden' : 'visible';
        }
    }
    //hide/show SOS rendered control portion
    if (name && isInputControl && !(controlElement && controlElement.classList.contains('equ-scrollable-area'))) {
        enableControl(name, !hidden);
    }
    // This is exception for scrollable area (scrollUp/Down buttons toggle visibility)
    if(controlElement && controlElement.classList.contains('equ-scrollable-area')){
        document.querySelector("#" + id + " .equ-scroll-up").style.visibility = hidden ? 'hidden' : 'visible';
        document.querySelector("#" + id + " .equ-scroll-down").style.visibility = hidden ? 'hidden' : 'visible';
        enableControl(id+'-up', hidden ? false : true);
        enableControl(id+'-down', hidden ? false : true);
    }
}

/**

 *  @description
 *  This function enable or disable control touch area.
 *
 *  @param {string} controlName - control name
 *  @param {boolean} enable - enable or disable control touch area
 *
 * @instance
 *
 */
function enableControl(controlName, enable) {
    logger.debug('enableControl: controlName={}, enable={}', controlName, enable);
    var statusCode = secure.enableControl(controlName, enable);
    if (statusCode != 0) {
        throw new EQError(eqErrors.ACTION_ENABLE_CONTROL_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function clear signature.
 *
 * @instance
 *
 */
function clearSignature() {
    logger.debug('clearSignature: params={}', signatureParams);
    if (signatureParams) {
        setControlValue(signatureParams.name, '');
    } else {
        throw new EQError(eqErrors.ACTION_CLEAR_SIGNATURE_ERROR, signatureParams);
    }
}

/**

 *  @description
 *	This function set expiration date.
 *
 *  @param {string} date - date in format mmyy
 *
 *  @example
 *  setExpirationDate('0218');
 *
 * @instance
 *
 */
function setExpirationDate(date) {
    logger.debug('setExpirationDate: date={}', date);
    var dateValue;
    if (typeof(date) === 'object') {
        //for backward compatibility with SDK 1.12.3 or older
        dateValue = date.expDate;
    } else {
        dateValue = date;
    }
    if (dateValue && dateValue.length == 4) {
        var mm = parseInt(dateValue.substr(0, 2), 10);
        if (mm < 1 || mm > 12) {
            throw new EQError(eqErrors.EXPIRATION_DATE_INVALID, {expirationDate: dateValue});
        } else {
            dateValue = dateValue.substr(2, 2) + dateValue.substr(0, 2);
            var statusCode = p2pe.setExpirationDate(dateValue);
            if (statusCode != 0) {
                throw new EQError(eqErrors.EXPIRATION_DATE_UNABLE_TO_SET, {statusCode: statusCode});
            }
        }
    } else {
        throw new EQError(eqErrors.EXPIRATION_DATE_INVALID, {expirationDate: dateValue});
    }
}

/**

 *  @description
 *	This function will bypass CVV.
 *
 *  @example
 *  bypassCVV2();
 *
 * @instance
 *
 */
function bypassCVV2() {
    // logger.debug('bypassCVV2');
    var res = p2pe.bypassCVV2();
    if (res == 0){
        // expect signal encryptedCVV2() to arrive
        logger.info("bypassCVV2: expect signal encryptedCVV2() to arrive");
    } else {
        // not TransArmorDUKPT => no signal will be sent
        logger.info("bypassCVV2: not TransArmorDUKPT => no signal will be sent");
    }
}

/**

 *  @description
 *	This function returns all control values that exist on the current page.
 *
 * @instance
 *
 */
function getControlValues() {
    var controlValues = secure.getInputValues();
    logger.debug('getControlValues: values={}', controlValues);
    return controlValues;
}

/**

 *  @description
 *	This function returns data by control name.
 *
 *  @param {string} name - control name
 *
 * @instance
 *
 */
function getControlValue(name) {
    var controlValue = secure.getInputValues(name);
    logger.debug('getControlValue: name={} value={}', name, controlValue);
    return controlValue;
}

/**

 *  @description
 *	This function set control value.
 *
 *  @param {string} name - control name
 *  @param {string} value - value to be set
 *
 * @instance
 *
 */
function setControlValue(name, value) {
    logger.debug('setControlValue: name={}, value={}', name, value);
    var statusCode = secure.setInputValue(name, value);
    if (statusCode != 0) {
        throw new EQError(eqErrors.SET_CONTROL_VALUE_ERROR, {statusCode: statusCode});
    }
}

/**

 *  @description
 *	This function returns all control values that exist on the current page with addition info if exist.
 *
 * @instance
 *
 */
function getControlValuesExt() {
    var controlValues = secure.getInputValues2();
    logger.debug('getControlValuesExt: values={}', controlValues);
    return controlValues;
}

/**

 *  @description
 *	This function returns data by control name with additional info.
 *
 *  @param {string} name - control name
 *
 * @instance
 *
 */
function getControlValueExt(name) {
    var controlValue = secure.getInputValues2(name)[name];
    logger.debug('getControlValueExt: name={} value={}', name, controlValue);
    return controlValue;
}

/**
 *  @description
 *	TODO missing description
 *
 * @instance
 *
 */
function processDivBoxUpdate(data) {
    divBoxXMLParser.parseXML(data);
}