#!/bin/bash

print_help()
{
    echo "Usage: run.sh app"
}

if [ "$#" -ne 1 ]
then
    print_help
    exit
fi

APP=$1

if [[ ! -x "$APP" ]]
then
    echo "Invalid application $APP"
    exit
fi

is_development_mode()
{
    test "$(cat /sys/kernel/secure/state)" == "2"
}

if is_development_mode
then
    # Generate core dumps.
    ulimit -c unlimited
    
    # Don't run the restart loop to avoid interfering with terminal.py scripts.
    ./${APP}
    exit
fi

# https://unix.stackexchange.com/questions/317492/list-of-kill-signals
# https://www.gnu.org/software/bash/manual/html_node/Exit-Status.html
# "When a command terminates on a fatal signal whose number is N, Bash uses the value 128+N as the exit status."
SIGILL=$((128+4))
SIGABRT=$((128+6))
SIGFPE=$((128+8))
SIGSEGV=$((128+11))

# Restart if program terminated abnormally.
RESTART_SIGNALS="${SIGILL} ${SIGABRT} ${SIGFPE} ${SIGSEGV}"

run=true
first_run=true
while $run; do
    if [[ $first_run = false ]]
    then
        export DEV_TRACER_STARTUP_MSG=" restarted"
    fi
    start_time=$(date +%s)
    ./${APP}
    rc=$?
    
    # if app exits immediately (within 5 seconds after start), then assume
    # there is an error in startup code and exit to avoid endless restart loop
    if [[ $(($(date +%s) - start_time)) -le 5 ]]
    then
        exit $rc
    fi
    
    run=false
    first_run=false
    for SIGNAL in ${RESTART_SIGNALS}
    do
        if [[ $rc -eq $SIGNAL ]]
        then
            run=true
            break
        fi
    done
done
